type GQLEdge<Node> = {
  cursor?: any;
  node: Node | null;
};

export type GQLEdges<Node> =
  | ReadonlyArray<GQLEdge<Node> | null>
  | null
  | undefined;

/**
 * Given an array of nullable edges with nullable nodes, this will return an
 * array of the non-null nodes dervied from the non-null edges.
 */
export function reduceToNonNullNodes<Node>(edges: GQLEdges<Node>): Node[] {
  return reduceToNonNullNodesWithEndCursor(edges).nodes;
}

export interface NonNullNodesWithEndCursor<T> {
  endCursor: string | null;
  nodes: T[];
}

/**
 * Given an array of nullable edges with cursors and nullable nodes, this will
 * return an object containing array of the non-null nodes derived from the
 * non-null edges as well as the last valid cursor (or null if no valid
 * cursors are found).
 */
export function reduceToNonNullNodesWithEndCursor<Node>(
  edges: GQLEdges<Node>,
): NonNullNodesWithEndCursor<Node> {
  let endCursor: string | null = null;
  const nodes = (edges ?? []).reduce<Node[]>((acc, edge) => {
    if (edge) {
      if (edge.cursor) {
        endCursor = edge.cursor;
      }
      if (edge.node) {
        acc.push(edge.node);
      }
    }

    return acc;
  }, []);

  return { endCursor, nodes };
}

interface TrackingEdge<Node> extends GQLEdge<Node> {
  trackingID: string;
}

type TrackingEdges<Node> =
  | ReadonlyArray<TrackingEdge<Node> | null>
  | null
  | undefined;

export type TrackingNode<Node> = Node & { trackingID: string };

/**
 * Given an array of nullable edges with trackingID's and nullable nodes, this will return an
 * array of the non-null nodes dervied from the non-null edges, with the
 * trackingID merged into the node.
 *
 */
export function reduceToNonNullNodesWithTrackingID<Node>(
  edges: TrackingEdges<Node>,
): TrackingNode<Node>[] {
  return (edges ?? []).reduce<TrackingNode<Node>[]>((acc, edge) => {
    if (edge?.node) {
      acc.push({ ...edge.node, trackingID: edge.trackingID });
    }
    return acc;
  }, []);
}
