import {
  reduceToNonNullNodes,
  reduceToNonNullNodesWithEndCursor,
  reduceToNonNullNodesWithTrackingID,
} from '.';

describe('reduceToNonNullNodes', () => {
  it('works on missing array', () => {
    expect(reduceToNonNullNodes(null)).toEqual([]);
  });

  it('works on empty array', () => {
    expect(reduceToNonNullNodes([])).toEqual([]);
  });

  it('works on array with all cases', () => {
    const list = [
      { node: null },
      { node: 1 },
      null,
      { node: 2 },
      { node: null },
      { node: 3 },
      null,
      { node: null },
    ];
    expect(reduceToNonNullNodes(list)).toEqual([1, 2, 3]);
  });
});

describe('reduceToNonNullNodesWithEndCursor', () => {
  it('works on missing array', () => {
    expect(reduceToNonNullNodesWithEndCursor(null)).toEqual({
      endCursor: null,
      nodes: [],
    });
  });

  it('works on empty array', () => {
    expect(reduceToNonNullNodesWithEndCursor([])).toEqual({
      endCursor: null,
      nodes: [],
    });
  });

  it('works on an array that has no valid cursors', () => {
    const list = [{ node: null }, { node: 1 }, null, { node: 2 }];
    expect(reduceToNonNullNodesWithEndCursor(list)).toEqual({
      endCursor: null,
      nodes: [1, 2],
    });
  });

  it('works on an array with cursors', () => {
    const list = [
      { cursor: '0', node: null },
      { cursor: '1', node: 1 },
      null,
      { cursor: '2', node: 2 },
    ];
    expect(reduceToNonNullNodesWithEndCursor(list)).toEqual({
      endCursor: '2',
      nodes: [1, 2],
    });
  });

  it('works on an array whose last valid cursor does not have a node', () => {
    const list = [
      { cursor: '0', node: null },
      { cursor: '1', node: 1 },
      null,
      { cursor: '2', node: 2 },
      { cursor: '3', node: null },
      null,
    ];
    expect(reduceToNonNullNodesWithEndCursor(list)).toEqual({
      endCursor: '3',
      nodes: [1, 2],
    });
  });
});

describe(reduceToNonNullNodesWithTrackingID, () => {
  it('works on missing array', () => {
    expect(reduceToNonNullNodesWithTrackingID(null)).toEqual([]);
  });

  it('works on empty array', () => {
    expect(reduceToNonNullNodesWithTrackingID([])).toEqual([]);
  });

  it('works on array with all cases', () => {
    const list = [
      { node: null, trackingID: '1' },
      { node: { val: 1 }, trackingID: '2' },
      null,
      { node: { val: 2 }, trackingID: '3' },
      { node: null, trackingID: '4' },
      { node: { val: 3 }, trackingID: '5' },
      null,
      { node: null, trackingID: '6' },
    ];
    expect(reduceToNonNullNodesWithTrackingID(list)).toEqual([
      { trackingID: '2', val: 1 },
      { trackingID: '3', val: 2 },
      { trackingID: '5', val: 3 },
    ]);
  });
});
