import {
  GLOBAL_COOKIE_NAMES,
  clearCookieValue,
  getAndExtendCookieValue,
  setCookieValue,
} from '../../cookies';
import { DEFAULT_ID_LENGTH } from '../uniqueIDGenerator';
import {
  TACHYON_DEVICE_ID_PREFIX,
  createTachyonDeviceID,
  getDeviceIDInBrowser,
  getOrCreateDeviceIDInBrowser,
} from '.';

describe('deviceID', () => {
  describe(createTachyonDeviceID, () => {
    it('generates a device id with the right prefix and length', () => {
      const id = createTachyonDeviceID();
      expect(id.startsWith(TACHYON_DEVICE_ID_PREFIX)).toBeTruthy();
      expect(id).toHaveLength(DEFAULT_ID_LENGTH);
    });
  });

  describe('browser methods', () => {
    afterEach(() => {
      clearCookieValue({ name: GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME });
    });

    describe(getDeviceIDInBrowser, () => {
      it('returns a device id cookie value when present', () => {
        const id = createTachyonDeviceID();
        setCookieValue({
          name: GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME,
          value: id,
        });

        expect(getDeviceIDInBrowser()).toEqual(id);
      });

      it('returns undefined when no matching cookie in request', () => {
        expect(getDeviceIDInBrowser()).toBeUndefined();
      });
    });

    describe(getOrCreateDeviceIDInBrowser, () => {
      it('generates a device id when none is present and stores it', () => {
        expect(
          getAndExtendCookieValue({
            name: GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME,
          }),
        ).toBeUndefined();
        const id = getOrCreateDeviceIDInBrowser();
        expect(
          getAndExtendCookieValue({
            name: GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME,
          }),
        ).toEqual(id);
      });

      it('retreives a device id that is previously stored and does not overwrite', () => {
        const id = createTachyonDeviceID();
        setCookieValue({
          name: GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME,
          value: id,
        });
        expect(getOrCreateDeviceIDInBrowser()).toEqual(id);
        expect(
          getAndExtendCookieValue({
            name: GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME,
          }),
        ).toEqual(id);
      });
    });
  });
});
