import type { FC } from 'react';
import { createContext, useCallback, useMemo, useRef } from 'react';
import { uniqueIDGenerator } from 'tachyon-utils-twitch';
import type {
  ChannelPageTrackingParams,
  PageviewMedium,
  PlayerTrackingParams,
  RecommendationTrackingParams,
  SearchClickResultParams,
  SearchClickUIParams,
  SearchInputClearParams,
  SearchQuerySubmitParams,
  SearchResultImpressionParams,
  SearchSuggestionDisplayParams,
  SearchSuggestionRequestParams,
  SearchSuggestionsClickParams,
  SpadeDiscoveryEvent,
  SpadeRecommendationItemClickEvent,
  SpadeRecommendationItemDisplayEvent,
  SpadeSearchInputClearEvent,
  SpadeSearchInputFocusEvent,
  SpadeSearchQuerySubmitEvent,
  SpadeSearchSuggestionClickEvent,
  SpadeSearchSuggestionDisplayEvent,
  SpadeSearchSuggestionRequestEvent,
} from '../../../utils/tracking';
import {
  RecommendationEventType,
  SearchEventType,
  recommendationParamsToBaseEvent,
} from '../../../utils/tracking';
import { channelPageParamsToBaseEvent } from '../../../utils/tracking/channelPageParamsToProperties';

export interface DiscoveryTrackingContext {
  getAndResetPlayerTracking: () => PlayerTrackingParams;
  getPlayerTracking: () => PlayerTrackingParams;
  onChannelPageItemClick: (params: ChannelPageTrackingParams) => void;
  onChannelPageItemDisplay: (params: ChannelPageTrackingParams) => void;
  onRecommendationItemClick: (params: RecommendationTrackingParams) => void;
  onRecommendationItemDisplay: (params: RecommendationTrackingParams) => void;
  onSearchInputClear: (params: SearchInputClearParams) => void;
  onSearchInputFocus: () => void;
  onSearchQuerySubmit: (params: SearchQuerySubmitParams) => void;
  onSearchResultClick: (params: SearchClickResultParams) => void;
  onSearchResultImpression: (params: SearchResultImpressionParams) => void;
  onSearchSuggestionClick: (params: SearchSuggestionsClickParams) => void;
  onSearchSuggestionDisplay: (params: SearchSuggestionDisplayParams) => void;
  onSearchSuggestionRequest: (params: SearchSuggestionRequestParams) => void;
  onSearchUIClick: (params: SearchClickUIParams) => void;
}

const noopPlayerTracking: Readonly<PlayerTrackingParams> = {
  item_tracking_id: undefined,
  medium: undefined,
  model_tracking_id: undefined,
  search_query_id: undefined,
  search_session_id: undefined,
};

// istanbul ignore next: trivial
export const noopDiscoveryTrackingContext: Readonly<DiscoveryTrackingContext> =
  {
    getAndResetPlayerTracking: () => {
      return noopPlayerTracking;
    },
    getPlayerTracking: () => {
      return noopPlayerTracking;
    },
    onChannelPageItemClick: () => {
      return;
    },
    onChannelPageItemDisplay: () => {
      return;
    },
    onRecommendationItemClick: () => {
      return;
    },
    onRecommendationItemDisplay: () => {
      return;
    },
    onSearchInputClear: () => {
      return;
    },
    onSearchInputFocus: () => {
      return;
    },
    onSearchQuerySubmit: () => {
      return;
    },
    onSearchResultClick: () => {
      return;
    },
    onSearchResultImpression: () => {
      return;
    },
    onSearchSuggestionClick: () => {
      return;
    },
    onSearchSuggestionDisplay: () => {
      return;
    },
    onSearchSuggestionRequest: () => {
      return;
    },
    onSearchUIClick: () => {
      return;
    },
  };

export const discoveryTrackingContext = createContext<DiscoveryTrackingContext>(
  noopDiscoveryTrackingContext,
);

export interface DiscoveryTrackingRootProps {
  currentRouteName: string;
  getMediumFromRoute: (routeName: string) => PageviewMedium | undefined;
  /**
   * For setting the player-attribution for `item_tracking_id` for when recs are consumed
   * externally and the app booted via deeplink directly to a player page
   */
  initialItemTrackingId?: string;
  /** For setting the player-attribution for `model_tracking_id` for when recs are consumed
   * externally and the app booted via deeplink directly to a player page
   */
  initialModelTrackingId?: string;
  onEvent: (event: SpadeDiscoveryEvent) => void;
  searchCorrelationRoutes: string[];
}

export const DiscoveryTrackingRoot: FC<DiscoveryTrackingRootProps> = ({
  children,
  currentRouteName,
  getMediumFromRoute,
  initialItemTrackingId,
  initialModelTrackingId,
  onEvent,
  searchCorrelationRoutes,
}) => {
  const searchQueryIdRef = useRef<string | undefined>();
  const searchSessionIdRef = useRef<string | undefined>();
  const mediumRef = useRef<PageviewMedium | undefined>();
  const itemTrackingIdRef = useRef<string | undefined>(initialItemTrackingId);
  const modelTrackingIdRef = useRef<string | undefined>(initialModelTrackingId);

  const onRecommendationItemClick = useCallback(
    (params: RecommendationTrackingParams) => {
      const baseEvent = recommendationParamsToBaseEvent(params);
      itemTrackingIdRef.current = baseEvent.item_tracking_id;

      const event: SpadeRecommendationItemClickEvent = {
        event: RecommendationEventType.ItemClick,
        ...baseEvent,
      };

      onEvent(event);
    },
    [onEvent],
  );

  const onRecommendationItemDisplay = useCallback(
    (params: RecommendationTrackingParams) => {
      const event: SpadeRecommendationItemDisplayEvent = {
        event: RecommendationEventType.ItemDisplay,
        ...recommendationParamsToBaseEvent(params),
      };

      onEvent(event);
    },
    [onEvent],
  );

  // convenience functions for clearing tracking values (not exported)
  const resetSearchTracking = useCallback(() => {
    searchQueryIdRef.current = undefined;
    searchSessionIdRef.current = undefined;
  }, []);

  const resetMediumTracking = useCallback(() => {
    itemTrackingIdRef.current = undefined;
    mediumRef.current = undefined;
    modelTrackingIdRef.current = undefined;
  }, []);

  const resetPlayerTracking = useCallback(() => {
    resetMediumTracking();
    resetSearchTracking();
  }, [resetMediumTracking, resetSearchTracking]);

  // search-specific callbacks
  const onSearchInputFocus = useCallback(() => {
    searchSessionIdRef.current = uniqueIDGenerator();
    const event: SpadeSearchInputFocusEvent = {
      event: SearchEventType.SearchInputFocus,
      search_session_id: searchSessionIdRef.current,
    };
    onEvent(event);
  }, [onEvent]);

  const onSearchInputClear = useCallback(
    ({ query }: SearchInputClearParams) => {
      const event: SpadeSearchInputClearEvent = {
        event: SearchEventType.SearchInputClear,
        query,
        search_query_id: searchQueryIdRef.current,
        search_session_id: searchSessionIdRef.current,
      };
      onEvent(event);
      resetSearchTracking();
    },
    [onEvent, resetSearchTracking],
  );

  // istanbul ignore next: trivial
  const onSearchSuggestionRequest = useCallback(
    ({ query, requestID }: SearchSuggestionRequestParams) => {
      const event: SpadeSearchSuggestionRequestEvent = {
        event: SearchEventType.SearchSuggestionRequest,
        query,
        request_id: requestID,
        search_session_id: searchSessionIdRef.current,
      };
      onEvent(event);
    },
    [onEvent],
  );

  // istanbul ignore next: trivial
  const onSearchSuggestionDisplay = useCallback(
    (params: SearchSuggestionDisplayParams) => {
      const event: SpadeSearchSuggestionDisplayEvent = {
        event: SearchEventType.SearchSuggestionDisplay,
        item_position: params.itemPosition,
        model_tracking_id: params.modelTrackingID ?? null,
        request_id: params.requestID,
        response_id: params.responseID ?? null,
        search_session_id: searchSessionIdRef.current,
        suggestion_text: params.suggestionText,
        suggestion_tracking_id: params.suggestionTrackingID,
        suggestion_type: params.suggestionType,
      };
      onEvent(event);
    },
    [onEvent],
  );

  // istanbul ignore next: trivial
  const onSearchSuggestionClick = useCallback(
    (params: SearchSuggestionsClickParams) => {
      const event: SpadeSearchSuggestionClickEvent = {
        clicked_item_id: params.clickedItemID,
        event: SearchEventType.SearchSuggestionClick,
        item_position: params.itemPosition,
        model_tracking_id: params.modelTrackingID ?? null,
        request_id: params.requestID,
        response_id: params.responseID ?? null,
        search_session_id: searchSessionIdRef.current,
        suggestion_text: params.suggestionText,
        suggestion_tracking_id: params.suggestionTrackingID,
        suggestion_type: params.suggestionType,
      };
      onEvent(event);
    },
    [onEvent],
  );

  const onSearchQuerySubmit = useCallback(
    (params: SearchQuerySubmitParams) => {
      searchQueryIdRef.current = uniqueIDGenerator();
      const event: SpadeSearchQuerySubmitEvent = {
        event: SearchEventType.SearchQuerySubmit,
        query: params.query,
        search_query_id: searchQueryIdRef.current,
        search_session_id: searchSessionIdRef.current,
        suggestion_position: params.suggestionPosition ?? null,
        suggestion_tracking_id: params.suggestionTrackingID ?? null,
      };
      onEvent(event);
    },
    [onEvent],
  );

  // istanbul ignore next: trivial
  const onSearchUIClick = useCallback(
    (params: SearchClickUIParams) => {
      onEvent({
        button_type: params.buttonType,
        event: SearchEventType.SearchUIClick,
        query: params.query,
        search_query_id: searchQueryIdRef.current,
        search_session_id: searchSessionIdRef.current,
        sub_section: params.subSection ?? null,
      });
    },
    [onEvent],
  );

  // istanbul ignore next: trivial
  const onSearchResultClick = useCallback(
    (params: SearchClickResultParams) => {
      onEvent({
        content_id: params.contentID ?? null,
        content_type: params.contentType,
        event: SearchEventType.SearchResultClick,
        game: params.gameID ?? null,
        is_live: params.isLive ?? false,
        item_position: params.itemIndex,
        query: params.query,
        search_query_id: searchQueryIdRef.current,
        search_session_id: searchSessionIdRef.current,
        show_all: params.showAll ?? false,
        show_more: params.showMore ?? false,
        srp_item_tracking_id: params.srpItemTrackingID ?? null,
        sub_section: params.subSection ?? null,
        sub_section_position: params.itemRow,
        vod_id: params.vodID ?? null,
      });
    },
    [onEvent],
  );

  // istanbul ignore next: trivial
  const onSearchResultImpression = useCallback(
    (params: SearchResultImpressionParams) => {
      onEvent({
        content_id: params.contentID ?? null,
        event: SearchEventType.SearchResultImpression,
        is_live: params.isLive ?? false,
        item_id: params.contentID ?? null,
        item_position: params.itemIndex,
        query: params.query,
        search_query_id: searchQueryIdRef.current,
        search_session_id: searchSessionIdRef.current,
        srp_item_tracking_id: params.srpItemTrackingID ?? null,
        sub_section: params.subSection,
        sub_section_position: params.itemRow,
      });
    },
    [onEvent],
  );

  const onChannelPageItemDisplay = useCallback(
    (params: ChannelPageTrackingParams) => {
      onEvent({
        ...channelPageParamsToBaseEvent(params),
        event: RecommendationEventType.ItemDisplay,
      });
    },
    [onEvent],
  );

  const onChannelPageItemClick = useCallback(
    (params: ChannelPageTrackingParams) => {
      const baseEvent = channelPageParamsToBaseEvent(params);
      itemTrackingIdRef.current = baseEvent.item_tracking_id;
      onEvent({
        ...channelPageParamsToBaseEvent(params),
        event: RecommendationEventType.ItemClick,
      });
    },
    [onEvent],
  );

  // We provide this to the context for debugging / testing purposes
  const getPlayerTracking = useCallback((): PlayerTrackingParams => {
    return {
      item_tracking_id: itemTrackingIdRef.current,
      medium: mediumRef.current,
      model_tracking_id: modelTrackingIdRef.current,
      search_query_id: searchQueryIdRef.current,
      search_session_id: searchSessionIdRef.current,
    };
  }, []);

  // for passing tracking to player
  const getAndResetPlayerTracking = useCallback((): PlayerTrackingParams => {
    const trackingParams = getPlayerTracking();
    resetPlayerTracking();
    return trackingParams;
  }, [getPlayerTracking, resetPlayerTracking]);

  // ensure a stable context value to prevent down-tree re-renders
  const discoveryTrackingContextValue = useMemo(
    () => ({
      getAndResetPlayerTracking,
      getPlayerTracking,
      onChannelPageItemClick,
      onChannelPageItemDisplay,
      onRecommendationItemClick,
      onRecommendationItemDisplay,
      onSearchInputClear,
      onSearchInputFocus,
      onSearchQuerySubmit,
      onSearchResultClick,
      onSearchResultImpression,
      onSearchSuggestionClick,
      onSearchSuggestionDisplay,
      onSearchSuggestionRequest,
      onSearchUIClick,
    }),
    [
      getAndResetPlayerTracking,
      getPlayerTracking,
      onChannelPageItemClick,
      onChannelPageItemDisplay,
      onRecommendationItemClick,
      onRecommendationItemDisplay,
      onSearchInputClear,
      onSearchInputFocus,
      onSearchQuerySubmit,
      onSearchResultClick,
      onSearchResultImpression,
      onSearchSuggestionClick,
      onSearchSuggestionDisplay,
      onSearchSuggestionRequest,
      onSearchUIClick,
    ],
  );

  // Search tracking gets reset if user enters a non-search route
  if (!searchCorrelationRoutes.includes(currentRouteName)) {
    resetSearchTracking();
  }

  // Medium and recommendations tracking gets reset when a route associated with a medium is entered
  const mediumFromRoute = getMediumFromRoute(currentRouteName);
  if (mediumFromRoute) {
    resetMediumTracking();
    mediumRef.current = mediumFromRoute;
  }

  return (
    <discoveryTrackingContext.Provider value={discoveryTrackingContextValue}>
      {children}
    </discoveryTrackingContext.Provider>
  );
};

DiscoveryTrackingRoot.displayName = 'DiscoveryTrackingRoot';
