import { datatype } from 'faker';
import type { PropsWithChildren } from 'react';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { uniqueIDGenerator } from 'tachyon-utils-twitch';
import {
  PageviewMedium,
  RecommendationEventType,
  SearchEventType,
} from '../../../utils/tracking';
import type { DiscoveryTrackingRootProps } from '.';
import { DiscoveryTrackingRoot, discoveryTrackingContext } from '.';

jest.mock('tachyon-utils-twitch', () => ({
  ...jest.requireActual('tachyon-utils-twitch'),
  uniqueIDGenerator: jest.fn(),
}));
const mockUniqueIdGenerator = uniqueIDGenerator as jest.Mock;

describe('DiscoveryTrackingRoot', () => {
  const MOCK_MEDIUM_FROM_ROUTE_MAP: { [key: string]: PageviewMedium } = {
    home: PageviewMedium.Discover,
    search: PageviewMedium.Search,
  };

  beforeEach(() => {
    mockUniqueIdGenerator.mockReset();
  });

  const ContextEchoer = (_props: any) => null;

  const setup = createMountWrapperFactory(
    DiscoveryTrackingRoot,
    (): PropsWithChildren<DiscoveryTrackingRootProps> => ({
      children: (
        <discoveryTrackingContext.Consumer>
          {(context) => <ContextEchoer {...context} />}
        </discoveryTrackingContext.Consumer>
      ),
      currentRouteName: 'home',
      getMediumFromRoute: (route: string) => {
        return MOCK_MEDIUM_FROM_ROUTE_MAP[route];
      },
      onEvent: jest.fn(),
      searchCorrelationRoutes: ['search'],
    }),
  );

  describe('recommendations tracking', () => {
    it('emits an item click event and stores the tracking id for attribution', () => {
      const { props, wrapper } = setup();
      const context = wrapper.find(ContextEchoer).props();

      const expectedItemTrackingID = datatype.uuid();

      context.onRecommendationItemClick({
        itemTrackingID: expectedItemTrackingID,
      });

      expect(props.onEvent).toHaveBeenCalledTimes(1);
      expect(props.onEvent).toHaveBeenCalledWith(
        expect.objectContaining({
          event: RecommendationEventType.ItemClick,
        }),
      );

      expect(context.getAndResetPlayerTracking()).toMatchObject({
        item_tracking_id: expectedItemTrackingID,
      });
    });

    it('emits an item display event', () => {
      const { props, wrapper } = setup();
      const context = wrapper.find(ContextEchoer).props();

      const expectedItemTrackingID = datatype.uuid();

      context.onRecommendationItemDisplay({
        item_tracking_id: expectedItemTrackingID,
      });

      expect(props.onEvent).toHaveBeenCalledTimes(1);
      expect(props.onEvent).toHaveBeenCalledWith(
        expect.objectContaining({
          event: RecommendationEventType.ItemDisplay,
        }),
      );
    });
  });

  describe('channel page tracking', () => {
    it('emits an item click event and stores the tracking id for attribution', () => {
      const { props, wrapper } = setup();
      const context = wrapper.find(ContextEchoer).props();

      const expectedItemTrackingID = datatype.uuid();

      context.onChannelPageItemClick({
        itemTrackingID: expectedItemTrackingID,
      });

      expect(props.onEvent).toHaveBeenCalledTimes(1);
      expect(props.onEvent).toHaveBeenCalledWith(
        expect.objectContaining({
          event: RecommendationEventType.ItemClick,
        }),
      );

      expect(context.getAndResetPlayerTracking()).toMatchObject({
        item_tracking_id: expectedItemTrackingID,
      });
    });

    it('emits an item display event', () => {
      const { props, wrapper } = setup();
      const context = wrapper.find(ContextEchoer).props();

      const expectedItemTrackingID = datatype.uuid();

      context.onChannelPageItemDisplay({
        item_tracking_id: expectedItemTrackingID,
      });

      expect(props.onEvent).toHaveBeenCalledTimes(1);
      expect(props.onEvent).toHaveBeenCalledWith(
        expect.objectContaining({
          event: RecommendationEventType.ItemDisplay,
        }),
      );
    });
  });

  describe('search tracking', () => {
    it('sets search session id when the search input is focused and emits an event', () => {
      const { props, wrapper } = setup();
      const context = wrapper.find(ContextEchoer).props();

      const expectedSessionID = datatype.uuid();
      mockUniqueIdGenerator.mockReturnValue(expectedSessionID);

      context.onSearchInputFocus();

      expect(
        wrapper.find(ContextEchoer).props().getPlayerTracking(),
      ).toMatchObject({
        search_query_id: undefined,
        search_session_id: expectedSessionID,
      });

      expect(props.onEvent).toHaveBeenCalledTimes(1);
      expect(props.onEvent).toHaveBeenCalledWith(
        expect.objectContaining({
          event: SearchEventType.SearchInputFocus,
        }),
      );
    });

    it('emits an event and sets a query id when a search result is submitted', () => {
      const { props, wrapper } = setup();
      const context = wrapper.find(ContextEchoer).props();

      const expectedQueryID = datatype.uuid();
      mockUniqueIdGenerator.mockReturnValue(expectedQueryID);

      context.onSearchQuerySubmit({ query: 'foo', time: 0 });

      expect(
        wrapper.find(ContextEchoer).props().getPlayerTracking(),
      ).toMatchObject({
        search_query_id: expectedQueryID,
        search_session_id: undefined,
      });

      expect(props.onEvent).toHaveBeenCalledTimes(1);
      expect(props.onEvent).toHaveBeenCalledWith(
        expect.objectContaining({
          event: SearchEventType.SearchQuerySubmit,
        }),
      );
    });

    it('resets search tracking ids when the input is cleared', () => {
      const { wrapper } = setup();
      const context = wrapper.find(ContextEchoer).props();

      const expectedID = datatype.uuid();
      mockUniqueIdGenerator.mockReturnValue(expectedID);

      context.onSearchInputFocus();
      context.onSearchQuerySubmit({ query: 'foo', time: 0 });

      expect(
        wrapper.find(ContextEchoer).props().getPlayerTracking(),
      ).toMatchObject({
        search_query_id: expectedID,
        search_session_id: expectedID,
      });

      context.onSearchInputClear({ query: 'foo' });

      expect(
        wrapper.find(ContextEchoer).props().getPlayerTracking(),
      ).toMatchObject({
        search_query_id: undefined,
        search_session_id: undefined,
      });
    });

    it('resets search tracking ids when a non-search route is entered', () => {
      const { wrapper } = setup();
      const context = wrapper.find(ContextEchoer).props();

      const expectedID = datatype.uuid();
      mockUniqueIdGenerator.mockReturnValue(expectedID);

      context.onSearchInputFocus();
      context.onSearchQuerySubmit({ query: 'foo', time: 0 });

      expect(
        wrapper.find(ContextEchoer).props().getPlayerTracking(),
      ).toMatchObject({
        search_query_id: expectedID,
        search_session_id: expectedID,
      });

      wrapper.setProps({ currentRouteName: 'homepage' });

      expect(
        wrapper.find(ContextEchoer).props().getPlayerTracking(),
      ).toMatchObject({
        search_query_id: undefined,
        search_session_id: undefined,
      });
    });
  });

  it('sets medium when an appropriate route is entered', () => {
    const { wrapper } = setup();
    const context = wrapper.find(ContextEchoer).props();

    expect(context.getAndResetPlayerTracking()).toMatchObject({
      medium: 'discover',
    });

    wrapper.setProps({ currentRouteName: 'search' });

    expect(context.getAndResetPlayerTracking()).toMatchObject({
      medium: 'search',
    });
  });

  it('resets all player tracking params when they are used for attribution', () => {
    const { wrapper } = setup();
    const context = wrapper.find(ContextEchoer).props();

    const expectedID = datatype.uuid();
    mockUniqueIdGenerator.mockReturnValue(expectedID);

    context.onSearchInputFocus();
    context.onSearchQuerySubmit({ query: 'foo', time: 0 });
    context.onRecommendationItemClick({
      item_tracking_id: expectedID,
      medium: PageviewMedium.Discover,
    });

    expect(context.getAndResetPlayerTracking()).toEqual({
      item_tracking_id: expectedID,
      medium: PageviewMedium.Discover,
      search_query_id: expectedID,
      search_session_id: expectedID,
    });

    expect(context.getAndResetPlayerTracking()).toEqual({
      item_tracking_id: undefined,
      medium: undefined,
      search_query_id: undefined,
      search_session_id: undefined,
    });
  });
});
