# Accessibility

## Resources

- [Accessibility in Core-UI](https://design.xarth.tv/user-interface/guidelines/accessibility)
- [How Screen Readers Read Content](https://webaim.org/techniques/screenreader/#how)

## Text To Speech (Screen Readers)

### Page Navigation Structure

Use semantic
[Content](https://developer.mozilla.org/en-US/docs/Web/HTML/Element#Content_sectioning)
elements to outline the page navigation hierarchy:

- Each page must have a
  [<nav>](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/nav) and
  [<main>](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/main)
  element.
- Secondary content on a page should be within an
  [<aside>](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/aside)
  element.
- Use
  [<section>](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/section)
  elements to indicate multiple related but different content areas.
- Set appropriate
  [heading](https://webaim.org/techniques/semanticstructure/#contentstructure)
  elements to indicate titles, not just visual cues such as sizing or bolding.

### Supplementing Visual Context With ARIA Labels

Some text labels only make sense in the visual context they are contained in. To
aid users reliant on screen readers, use `aria-label` to provide the missing
context. Examples include:

- Viewer / Streamer / Follower counts
- SVG-only buttons
- Tags

### ARIA Live Regions for Dynamic Page Content

See
[ARIA Live Regions](https://developer.mozilla.org/en-US/docs/Web/Accessibility/ARIA/ARIA_Live_Regions).

### Images

Standalone images without a descriptive text heading must set a localized `alt`
attribute for screen readers to provide the missing visual context. Images that
have a corresponding text heading should set `alt=""` to indicate to the screen
reader that the image should be ignored.

### Hiding Critical Functionality

Ensure that critical page functionality
[can be read by a screen reader](https://webaim.org/techniques/css/invisiblecontent/#techniques)
even if it can be visibly hidden for sighted users.

Examples:

- hiding Player Controls after some duration of no user input activity

Avoid:

- `display: none;`
- `visibility: hidden;`
- `height: 0;`
- `width: 0;`
- `opacity: 0;`

### Advanced Screen Reader Compatible UI Patterns

#### [Media Card](https://design.xarth.tv/user-interface/patterns/card/media-cards)

Provided Semantics:

- `MediaCard` outputs markup wrapped in an
  [<article>](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/article).
- `MediaCardTitle` renders the primary card content title with an `<h3>`
  heading.
- `MediaCardLink` renders link text without a heading

Rules:

- The "image" prop provided to `MediaCardImage` should include an empty `alt=""`
  since `MediaCardTitle` provides a better context heading.

#### Shelves / Carousels

- Use a `<section>` to represent each shelf.
- The shelf's title should use a heading level relative to it's position in the
  page's content hierarchy. Default to `<h2>` if there is no higher level
  heading.
- Shelf entries should be contained within an unordered list to indicate that
  they are related

Example:

```tsx
<section>
  <h2>Popular Channels</h2>
  <ul>
    <li><MediaCard /></li>
    <li><MediaCard /></li>
    <li><MediaCard /></li>
  </ul>
</section>

<section>
  <h2>Popular Games</h2>
  <ul>
    <li><MediaCard /></li>
    <li><MediaCard /></li>
    <li><MediaCard /></li>
  </ul>
</section>

<...>
```

#### Tag Lists

```tsx
<div role="list">
  <Tag label="FPS" />
  <Tag label="Competitive" />
  <Tag label="Chill Stream" />
</div>
```

#### Infinite Scrolling

Follow this guide on using ARIA
[role="feed"](https://developer.mozilla.org/en-US/docs/Web/Accessibility/ARIA/Roles/Feed_Role).

```tsx
<section role="feed" aria-busy="false">
  ...
  <article aria-posinset="427" aria-setsize="-1">
    ...
  </article>
  <article aria-posinset="428" aria-setsize="-1">
    ...
  </article>
  <article aria-posinset="429" aria-setsize="-1">
    ...
  </article>
  ...
</section>
```

- aria-busy should be set to true when we are loading or removing items from the
  DOM.
- aria-posinset is the same as the item's index.
- aria-setsize should be used when we know the total size of the list of items.
  Generally this should be set to "-1".

## Testing Accessibility

Screen Reader Commands:

- OSX: `CMD + F5`
- Windows: `Win + Ctrl + Enter` built in screen readers.

Dev Tools:

- [Axe For Chrome](https://chrome.google.com/webstore/detail/axe-web-accessibility-tes/lhdoppojpmngadmnindnejefpokejbdd?hl=en-US)
