# Page Creation In Tachyon Apps

This guide will walk you through creating a Next.js
[Page](https://nextjs.org/docs/basic-features/pages) within a Tachyon app.

For this example, we'll assume that we want to create a new page that
dynamically fetches a Twitch channel with a navigable path of
`/[login]/somepage` (i.e. `/ninja/somepage`).

## Creating A Page Component

In the app's `src/components/pages` directory, add the following folders and
files:

```
pages/
  [login]/
    index.tsx
    test.ts
```

In `[login]/index.tsx`, we'll scaffold out a Page Component:

```tsx
import { TachyonPage } from 'tachyon-next-types';

// Pages should use an app specific Page type in src/pages/types.ts if one is present.
// Otherwise use TachyonPage.
export const SomePage: TachyonPage = () => <div />;
```

## Registering The Page With Next's Built-In Routing

Now we'll register the page component with Next's
[file-system based router](https://nextjs.org/docs/routing/introduction). In
`src/pages`, add an index file that adds a default re-export of the page
component you just created at `src/pages/[login]/index.ts`:

```ts
export { Somepage as default } from '../components';
```

At this point you should be able to navigate to your page in the application.

## Registering The Page With Tachyon's Route Link Builder

Tachyon apps leverage Typescript to ensure clean and consistent formation of
urls and query params. In order to take advantage of this infrastructure, you
must register each page with the routing system. First, add your page to the
`RouteName` enum in `src/routing/routes/index.ts`:

```ts
export enum RouteName {
  Somepage = 'somepage_name',
}
```

Next, in the same file, we'll need to add the pathname to our `RouteMap`:

```ts
export const RouteLinkPathnames: RouteMap = {
  [RouteName.Somepage]: '/[login]',
};
```

Finally, add the route to the `RouteLinkBuilders` object:

```ts
export const RouteLinkBuilders = {
  [RouteName.Somepage]: ({ login }: { login: string }): RouteLinkParams => ({
    as: `/${login}`,
    href: RouteLinkPathnames[RouteName.Somepage],
  }),
};
```

## Page Data Access With Relay

Tachyon apps request Twitch data through GraphQL using
[Relay](https://relay.dev/). Each page is responsible for defining its data
requirements in a single GraphQL query in order to remain SSR compatible.

**IMPORTANT:** If an app has page caching user specific data must be deferred to
client initialization to avoid cache poisoning. Reach out to the EP team for
specifics of this approach.

### Setting Up A Page Relay Query

1. Add a static GraphQL "query" to the page as seen below.
1. Run `yarn relay` in the application to generate specific types for the page
   query.
1. Add the generated query result type to the page's prop type.

```tsx
import { TachyonPage } from 'tachyon-next-types';
import { SomePage_QueryResponse } from './__generated__/SomePage_Channel.graphql';

export type SomePageProps = SomePage_QueryResponse;

export const SomePage: TachyonPage<{}, SomePageProps> = ({ channel }) => {
  // Consume the Relay data
};

SomePage.query = graphql`
  query SomePage_Channel {
    channel: user(login: "lirik") {
      login
    }
  }
`;
```

### Using getInitialProps To Provide Query Variables

To make our page dynamic, we'll need to provide the query with the "login" path
part to retrieve data specific to that path part. We'll add a "login" query
variable as seen below:

```tsx
SomePage.query = graphql`
  query SomePage_Query($login: String!) {
    channel: user(login: $login) {
      # ...
    }
  }
`;
```

Re-run `yarn relay` to generate the new `SomePage_QueryVariables` type. Now add
a
[getInitialProps](https://nextjs.org/docs/api-reference/data-fetching/getInitialProps)
static function to the page. This is where query variables must be set and is
also where the page will have access to the request object to access path /
query variables:

```tsx
import { TachyonPage } from 'tachyon-next-types';
import {
  SomePage_QueryVariables,
  SomePage_QueryResponse,
} from './__generated__/SomePage_Query.graphql';

type SomePageInitialProps = {
  queryVariables: SomePage_QueryVariables;
};

type SomePagePathParameters = { login: string };

export type SomePageProps = SomePageInitialProps & SomePage_QueryResponse;

export const SomePage: TachyonPage<
  SomePageInitialProps,
  SomePageProps,
  SomePagePathParameters
> = ({ channel }) => {
  /* ...*/
};

// The TachyonPage type with specified generics will provide type inference and enforce typings here
SomePage.getInitialProps = ({ query: { login } }) => {
  return {
    queryVariables: {
      login,
    },
  };
};
```

### Reading Object IDs from Path Parameters

For any ID values that come from the URL (e.g. VOD IDs), they will need to be
converted to a "safe" ID before using with Relay (in order to ensure ID
uniqueness in the Relay store). This can be done with one of the helper methods
exported from `tachyon-relay`:

```tsx
import { convertToSafeVideoID } from 'tachyon-relay';

type SomeVODPagePathParameters = { videoId: string };

const SomeVODPage: TachyonPage<{}, {}, SomeVODPagePathParameters> = () => {};

SomeVODPage.getInitialProps = ({ query }) => {
  const safeVideoID = convertToSafeVideoID(query.id);
  // Do something interesting with the Video ID.
};
```

Beyond that, the application opaquely handles the rest of the ID conversions
behind the scenes. If you need need a new convenience method for an unaddressed
type or run into problems with things like polymorphic types, please contact the
Mobile Web Platform team for assistance.

### Reading Values from URL Query Parameters

Next will place both path and query parameter values in the `query` value
provided in the `getInitialProps` argument:

```tsx
import { flattenHeaderOrParam } from 'tachyon-utils';

SomePage.getInitialProps = ({ query }) => ({
  someValueFromQueryParam: flattenHeaderOrParam(
    query['<SOME_URL_QUERY_PARAM_KEY>'],
  ),
});
```

## Checking For Nullable Objects

Due to being reliant on `getInitialProps`, our Relay setup waits for a
successful query response before proceeding with rendering the child component
it is provided. As a result, your component does not need to deal with loading
states.

**IMPORTANT: Twitch's GQL implementation is designed such that it rarely fails
and will instead return specific `null` fields if a downstream dependency has
reduced availability. This is indicated in our GQL Schema via Nullable fields
(those without a `!`) and will be identified as something you will need to
account for by Tachyon's use of
[Strict Null Checks](https://basarat.gitbooks.io/typescript/docs/options/strictNullChecks.html).
**

We use a
[User Defined Type Guard](https://www.typescriptlang.org/docs/handbook/advanced-types.html#user-defined-type-guards)
function to both determine whether a 404 experience should be rendered and to
narrow the type to be non-nullable for consumption within the component past
that point.

```tsx
import { TachyonPage } from 'tachyon-next-types';
import {
  SomePage_QueryVariables,
  SomePage_QueryResponse,
} from './__generated__/SomePage_Query.graphql';

export type SomePageProps = SomePage_QueryResponse;

function channelIsFound(
  channel: SomePage_QueryResponse['channel'],
): channel is NonNullable<SomePage_QueryResponse['channel']> {
  return !!channel;
}

export const SomePage: TachyonPage<SomePageInitialProps, SomePageProps> = ({
  channel,
}) => {
  if (!channelIsFound(channel)) {
    // 404 experience
  }

  // Null checks on channel are not required past this point due to the type guard above
};

SomePage.query = graphql`
  query SomePage_Channel {
    channel: user(login: "lirik") {
      login
    }
  }
`;
```

_Note: Our Relay setup does not currently expose any sort of error state
returned in our GQL response payload._

## Using Page-Utils For Common Page Types

Many common page level configuration and validation functions for Channel,
Category, Clip, VOD, and other pages have been made re-usable in the
[Page-Utils](https://git.xarth.tv/emerging-platforms/tachyon/tree/main/packages/tachyon-core/page-utils)
package including most of the logic in this tutorial!

## Pageview Tracking

A `pageview` event will be automatically emitted the first time a page is
rendered in any of our Tachyon apps. The `location` event field will be
automatically determined by the application's path to location mapping in the
app's routing layer.

Each Tachyon app page must define a static `pageviewTracking` function to ensure
that the page is providing the correct tracking event fields for its location.

For pages that do not need to populate additional `pageview` fields, default
pageview tracking can be used:

```tsx
import { defaultPageviewTracking } from 'tachyon-page-utils';

SomePage.pageviewTracking = defaultPageviewTracking;
```

Pages that are required to populate additional `pageview` fields can use one of
the existing utilities from `tachyon-page-utils`, or define their own mapping
function:

```tsx
SomePage.pageviewTracking = ({ channel }) => {
  if (!channelIsFound(channel)) {
    return {};
  }

  return {
    channel: channel.login,
  };
};
```

## Consuming Relay Data In Page Feature Components

Rather than passing resolved page data down directly to rendered feature
components, Relay requires a
[different pattern](https://relay.dev/docs/en/thinking-in-relay).

For more details see
[Data Access in Feature Components](./feature-components.md#data-access).

## GQL Partial Success Handling

The majority of new application pages have complex data requirements and should
be hardened against failing to render or behave correctly when non-critical data
is missing from the page's GQL query response. Follow this
[guide](../processes/gql-partial-success-tuning.md) to enable and tune your page
to handle these cases.
