# Custom React Hooks in the Tachyon Repo

[Custom Hooks](https://reactjs.org/docs/hooks-custom.html) are available as of
React 16.8. Before creating a new hook, check if we've already built it in our
[utils-hooks](../packages/tachyon-core/utils-hooks/README.md) package.

## Writing Custom Hooks

We recommend that all values passed into the hook are provided as a single
"props" parameter object. This greatly simplifies testing.

```tsx
import { useEffect, useState } from 'react';

export type UseCustomHookProps = {
  callback: (callCount: number) => void;
  someValue: string;
};

/**
 * Custom effect for loading / autoplaying content via MediaPlayer including
 * managing content changes.
 */
export function useCustomHook({
  callback,
  someValue,
}: UseCustomHookProps): void {
  const callCountRef = useRef(0);
  useEffect(() => {
    callback(callCountRef.current);
    callCountRef.current++;
  }, [someValue]);
}
```

## Testing Custom Hooks

We use
[@testing-library/react-hooks](https://github.com/testing-library/react-hooks-testing-library)
to test our custom hooks. Here are some examples of how to setup tests. As
mentioned above, taking a single "prop" object for all data vastly simplifies
the testing approach as seen below:

```tsx
import { renderHook } from '@testing-library/react-hooks';
import { useCustomHook, UseCustomHookProps } from '.';

describe(useCustomHook, () => {
  function mockProps(
    overrides?: Partial<UseCustomHookProps>,
  ): UseCustomHookProps {
    return {
      callback: jest.fn(),
      someValue: 'foo',
      ...overrides,
    };
  }

  it('invokes the callback with the call count', () => {
    const initialProps = mockProps();
    const { rerender } = renderHook(p => useCustomHook(p), {
        initialProps,
    });

    expect(initialProps.callback).toHaveBeenCalledTimes(1);
    expect(initialProps.callback).toHaveBeenCalledWith(0);

    rerender();
    expect(initialProps.callback).toHaveBeenCalledTimes(1);

    rerender({ ...initialProps, someValue: 'boop' });
    expect(initialProps.callback).toHaveBeenCalledTimes(2);
    expect(initialProps.callback).toHaveBeenCalledWith(1);
  });

  it('returns an expected value', () => {
    const initialProps = mockProps();
    const { result } = renderHook(p => useCustomHook(p), {
        initialProps,
    });

    expect(result).toEqual([...]);
  });
});
```

Note: There is currently an
[RFC](https://github.com/testing-library/react-hooks-testing-library/issues/56)
that might change this public API of this testing library to make it easier to
work with.

### Testing Custom Hooks With Mock Contexts

The `renderHook` function accepts a `wrapper` value as part of the optional
second argument object which can be used to provide a mock context value:

```tsx
it('reads a value from context', () => {
  const wrapper = ({ children }) => (
    <TestContext.Provider value="bar">{children}</TestContext.Provider>
  );

  const { result } = renderHook(() => useContext(TestContext), { wrapper });
  expect(result.current).toBe('bar');
});
```
