# Best Practices in Cypress

## Basics

Tachyon uses [Cypress]
(https://docs.cypress.io/guides/core-concepts/introduction-to-cypress) as its
testing framework for end-to-end testing. This guide will cover general testing
patterns and an overview on how we use Cypress.

Refer to https://docs.cypress.io/api/table-of-contents for searching other
Cypress-related issues.

Cypress prevents you from having to rely on raw Javascript and DOM
functionality, and bundles `chai-jquery`
(https://www.chaijs.com/plugins/chai-jquery/) and other useful extensions for
you to work with instead.

Every time you start writing a new suite of tests, wrap it in a `describe` block
and the `it` block is the actual block where tests are written (the same pattern
as in Jest, both inherited from Mocha):

```ts
describe("test suite description", () => {
  it('test block description', () => {
    expect(...).to.equal(true);
  });
});
```

To avoid clashing of global variables during runtime, we import explicitly from
`local-cypress`:

```ts
import { before, cy, expect, it } from 'local-cypress';
```

When dealing with multiple `describe` and `it` blocks in the same test suite, if
we are always executing the same initial operations, you can combine the tests
such that the common commands aren't executed multiple times:

```ts
describe('how to avoid unnecessary repetitions', () => {
  beforeEach(() => {
    cy.visit('https://git.xarth.tv');
  });

  it('first it block', () => {
    cy.get('h1');
  });

  it('second it block', () => {
    cy.get('h2');
  });
});
```

## Cypress Testing Style

### Running the App

When solely developing tests for existing functionality, use the prodlike build
(`yarn prodlike`) for faster performance and more stability from the
application. Otherwise if you expect to have to change the application as you
develop the tests, use the dev build (`yarn start`).

### Selectors

Targeting elements for assertions frequently can rely on normal CSS selectors.
In order to avoid having these tests become too synthetic, you should use things
that are relevant to a human user. The best is to target stable text on the page
which would be visual to users. As a fallback, using features that would be
detectable by screen-readers and other accessibility technology (eg like form
autofill) is also a good strategy as these are also relevant to users. This can
look like targetting ARIA labels, semantic input types (`input[type=password]`),
and even just semantic elements on a page (eg `main`, `nav`, or `section`). As a
last resort, we also provide the `getSc` and `findSc` helpers, which are
detailed below.

### Aliasing

Cypress allows for aliasing using `.as()` which can later be referenced in
`cy.get()` and `cy.wait()`:

```ts
cy.get('section').as('firstRow').should('have', 'something');
...
cy.get('@firstRow').should('have', 'something-new');
```

```ts
cy.intercept('https://example.com').as('example');
...
cy.wait('@example');
```

### Assertions

Asserting on Cypress objects/chains relies on the `should()` function:

```ts
cy.get('section').children().should('have.length', 5);
```

Asserting on other values relies on the `expect()` function:

```ts
expect([...]).to.have.length(5);
```

Both styles ultimately rely on the same BDD style assertions, just with slightly
different invocations.

### General Helpers

Helpers which can be used anywhere in Tachyon are written in the
`tachyon-cypress` package. Currently we have `getSc` and `findSc`, which operate
the same as the built-in Cypress functions `cy.get()` and `cy.find()` except
they allow you to select on styled-component display names:

```ts
cy.getSc('ScPageButton');
cy.get('div').findSc('ScScrollContainer');
```

### Starshot Helpers

Helpers created specifically for Starshot are written in
`cypress/support/command.ts` in the `starshot/` app directory. They are meant to
abstract the complexity of interaction with various remote-based inputs and how
they are handled by `tachyon-tv-nav`.

#### tvNav

Used for navigating around the page, simulating the arrow buttons:

```ts
cy.tvNav('right');
```

#### tvEnter

Used for selection, simulating the enter button:

```ts
cy.tvEnter();
```

#### tvScroll

Used to scroll up and down a page, simulating a wheel input:

```ts
cy.tvScroll('up');
```

### Testing for Events Sent

We use `cy.intercept()` for spying and stubbing network requests and responses.
Always use `cy.intercept()` before `cy.visit()` as otherwise we are unable to
intercept the call since it is already in progress. You can then wait for the
intercept by calling `cy.wait()` on the intercept through aliasing:

```ts
cy.intercept('https://gql.twitch.tv/gql').as('gqlUrl');
cy.visit('/');
cy.wait('@gqlUrl');
```

**in development** `cy.getSpadeEvent()` which would verify the event took place
and return intercepted event

### Waiting

Avoid using arbitrary `cy.wait()` periods. Instead, Cypress offers timeouts as
an optional parameter to functions like `cy.get()` and `cy.find()` where we wait
till the assertion passes. This allows dynamic wait times (Cypress polls during
the timeout window) and also makes it more obvious what the slow action is:

```ts
cy.find('ScScrollContainer', { timeout: 1000 });
```

## Tips on Writing Commands

When creating your own commands, you can hide the inner implementation of the
commands created by not logging them as shown below:

```ts
cy.get('body', { log: false });
```
