# Feature Component Patterns in the Tachyon Repo

This section covers common patterns for creating application features. Before
getting started you should also read up on Tachyon's
[naming conventions](./naming-conventions.md) and patterns around
[React Components](./react-components.md) and [testing](./unit-testing.md).

## Common Feature Functionality

Tachyon has a number of in-repo [packages](../packages/README.md) that provide
most app and feature level capabilities that are safe to use in universal
applications:

- [Custom Styling](./styling-components.md)
- [Internationalization](../packages/tachyon-core/intl/README.md#formatting-app-strings-for-internationalization)
- [Event Tracking](../packages/tachyon-core/event-tracker/README.md)
- [Latency Tracking](../packages/tachyon-core/latency-tracker/README.md)
- [Next App Routing & Query Param Access](../packages/tachyon-core/next-routing-utils/README.md)
- [Client Environment Info](../packages/tachyon-core/environment/README.md)
- [Logging](../packages/tachyon-core/logger/README.md)
- [Experiments](../packages/tachyon-core/experiments/README.md)
- [General Utilities](../packages/tachyon-core/utils/README.md)
- [Custom React Hooks](./custom-hooks.md)

Reach out in #mobile-web if you believe a common capability is missing.

## Data Access

When breaking down complex sets of functionality on a page, it is often helpful
to separate responsibilities at the feature level. Relay has built-in support
for this through the
[useFragment hook](https://relay.dev/docs/api-reference/use-fragment/) pattern
which allows components to declare their exact data requirements.

It is completely normal, and highly encouraged, to
[compose fragments](https://relay.dev/docs/guided-tour/rendering/fragments/#composing-fragments)
in order to separate concerns at various levels in your component hierarchy.

Fragments are consumed as part of
[each page's query in which they are referenced](https://relay.dev/docs/guided-tour/rendering/fragments/#composing-fragments-into-queries).
This means that all necessary data for a page is queried and resolved at once
alleviating the need for in-page, per-feature component loading states. Feature
components are still responsible for handling errors and partial successes.

### Declaring Static Data Requirements with the useFragment Hook

For the case where your data remains static for the duration of a page visit,
use the [useFragment hook](https://relay.dev/docs/api-reference/use-fragment/):

```tsx
import type { FC } from 'react';
import { graphql, useFragment } from 'react-relay/hooks';
import { SomeFeature_stream$key } from './__generated__/SomeFeature_stream.graphql';

const someFeatureFragment = graphql`
  fragment SomeFeature_stream on Stream {
    id
    broadcaster {
      login
    }
  }
`;

type SomeFeatureProps = {
  stream: SomeFeature_stream$key;
};

export const SomeFeature: FC<SomeFeatureProps> = ({ stream: streamRef }) => {
  const stream = useFragment(someFeatureFragment, streamRef);
  // ...
};
```

We generally name the prop for the fragment reference as `foo` instead of
`fooRef` for convenience, and then use a destructured alias for a temporary
rename before passing the value to `useFragment`. This minimizes the cognitive
overhead dealing with ref vs value for most users (and also eases the migration
pattern since parent components will not need to be changed as we convert from
legacy Relay containers to the newer hooks).

### Declaring Dynamic Data Requirements with Refetch Containers

In the case of a feature that has dynamic data (refreshable, pagination,
filtering, etc) use the
[useRefetchableFragment hook](https://relay.dev/docs/api-reference/use-refetchable-fragment/):

```tsx
import type { FC } from 'react';
import { graphql, useRefetchableFragment } from 'react-relay/hooks';
import { DynamicFeature_user$key } from './__generated__/DynamicFeature_user.graphql';

const dynamicFeatureFragment = graphql`
  fragment DynamicFeature_user on Query
  @refetchable(queryName: "DynamicFeature_RefetchQuery") {
    user($login: String!) {
      id
      broadcaster {
        login
      }
    }
  }
`;

type DynamicFeatureProps = {
  user: DynamicFeature_user$key;
}

export const DynamicFeature: FC<DynamicFeatureProps> = ({ user: userRef }) => {
  const [user, refetch] = useRefetchableFragment(dynamicFeatureFragment, userRef);

  function getNewUser() {
    refetch({
      login: ...
    });
  }

  return (
    <>
      <button onClick={getNewUser}>Reload</button>
      {/* ... */}
    </>
  );
};
```

When calling `refetch`, you only need to include the variables that need to
change. Any omitted variables will fall back to using the original query's
values. This means that `refetch({})` is equivalent to making a new request with
all of the original variables.

In addition to including the fragment in the parent queries in which it appears,
the Relay compiler will also synthesize a separate query for the refetch
operations; this query will be named according to the `queryName` value passed
to the `@refetchable` directive.

_Note: Currently this pattern only supports refetching fragments on Query due to
our GraphQL server not supporting the `Node` interface._

#### Suspense and Refetching

When a refetch happens, the component may suspend while it awaits data. This
means that you will need to include a Suspense boundary above the component to
catch this suspension and render a loading state (currently we use the
`DangerousServerSuspense` component from `tachyon-relay` to enable SSR).

```tsx
import { DangerousServerSuspense } from 'tachyon-relay';
import { DynamicFeature } from './DynamicFeature';
import { MyLoader } from './MyLoader';

const Page: FC = () => {
  ...

  return (
    <DangerousServerSuspense fallback={<MyLoader />}>
      <DynamicFeature ... />
    </DangerousServerSuspense>
  );
}
```

Once Concurrent mode is ready in a future version, React will allow suspending
while showing the stale tree. If you want to simulate this behavior until then,
you can use the `useRefetchableFragmentWithoutSuspense` hook from
`tachyon-relay` to create the same effect. It has the same API as
`useRefetchableFragment` but it additionally takes a full GraphQL query which
contains the target fragment. This pattern is still experimental so please ask
the Emerging Platforms team for assistance if you run into issues with it.

#### Directives and Refetching

The [@skip](https://relay.dev/docs/glossary/#skip) and
[@include](https://relay.dev/docs/glossary/#include) directives can be used for
refetchable fragments that are reliant on user interaction to populate query
variables (such as populating lookahead results as a user types, which would
only need to render after the component). In these situations, you might want to
skip some fields during the initial render and then include them only in refetch
queries:

```tsx
const inputDynamicFragment = graphql`
  fragment InputDynamic_user on Query
  @argumentDefinitions(
    login: { type; "String!" }
    noUser: { type: "Boolean!" }
  )
  @refetchable(queryName: "InputDynamic_RefetchQuery") {
    @skip(if: $noUser) {
      user($login: String!) {
        id
        broadcaster {
          login
        }
      }
    }
  }
`;

...

refetch({
  login: ...,
  noUser: false
});
```

Note that in these situations there will need to be an
[@argumentDefinitions](https://relay.dev/docs/api-reference/graphql-and-directives/#argumentdefinitions)
directive to provide Relay the proper context for synthesizing a refetch query
(since the directive variables do not originate from the GraphQL schema).
Similarly, there will need to be a matching
[@arguments](https://relay.dev/docs/api-reference/graphql-and-directives/#arguments)
defined in the parent query.

### Infinite lists with Refetch Containers

Our GraphQL backend does not support
[Connections](https://facebook.github.io/relay/graphql/connections.html) so we
are currently unable to utilize Relay's
[usePaginationFragment hook](https://relay.dev/docs/api-reference/use-pagination-fragment/).
Instead we use our custom `useRefetchList` hooks to simulate this behavior.

_Note: For now this example uses the older
[Refetch Container](https://relay.dev/docs/v10.1.3/refetch-container/). We are
working to update it to a newer hook pattern._

```tsx
import { FC, useCallback } from 'react';
import { createRefetchContainer, graphql, RelayRefetchProp } from 'react-relay';
import { useRefetchList } from 'tachyon-relay';

const CHANNELS_PAGE_SIZE = 10;

type ChannelsBaseProps = {
  query: Channels_query;
  relay: RelayRefetchProp;
};

export const ChannelsBase: FC<ChannelsBaseProps> = ({ query, relay }) => {
  const { game } = query;
  const {
    endCursor,
    noMore,
    nodes: streams,
  } = useRefetchList(query.game?.streams?.edges);

  const loadMore = useCallback(() => {
    if (noMore) {
      return;
    }

    relay.refetch(() => ({
      channelsCount: CHANNELS_PAGE_SIZE * 3,
      cursor: endCursor,
    }));
  }, [relay, endCursor, noMore]);

  const itemRenderer = useCallback(
    (idx: number): JSX.Element => (
      <SomeCard key={streams[idx].id} stream={streams[idx]} />
    ),
    [streams],
  );

  return (
    <InfiniteList
      itemRenderer={itemRenderer}
      length={streams.length}
      loadMore={loadMore}
      pageSize={CHANNELS_PAGE_SIZE}
    />
  );
};

ChannelsBase.displayName = 'ChannelsBase';

export const Channels = createRefetchContainer(
  ChannelsBase,
  {
    query: graphql`
      fragment Channels_query on Query {
        game(name: $gameAlias) {
          streams {
            edges {
              cursor
              node {
                id
                ...SomeCard_stream
              }
            }
          }
        }
      }
    `,
  },
  graphql`
    query Channels_RefetchQuery($gameAlias: String!, $cursor: Cursor) {
      ...Channels_query
    }
  `,
);
```

### Filtering Connection Nodes

Utilize our common utility functions for filtering nullable nodes in a GQL
connection including end cursor determination:

```ts
import {
  reduceToNonNullNodes,
  reduceToNonNullNodesWithEndCursor,
} from 'tachyon-utils';
```

### Accessing User Request Info

GraphQL's RequestInfo is only available on the client (since otherwise you'd get
cached information based on the server's IP address), and so should only be
consumed via the useRequestInfo hook. It will return `null` on the server and on
the client before the first app mount, and after that will expose all of the
requestInfo values. You must handle both the null and hydrated states in such a
way as to not cause reflow.

```tsx
import type { FC } from 'react';
import { useRequestInfo } from 'tachyon-relay';

export const RequestInfoConsumer: FC = () => {
  const requestInfo = useRequestInfo();

  if (requestInfo?.fromEU) {
    ...
  }

  return {
    /* ... */
  };
};
```

## Logging

In Tachyon apps, log events are sent to CloudWatch on the server, and
[Sentry](https://sentry.io/) on the client. Typically logging should be reserved
for notifying of unexpected edge cases only. When logging ensure that you
include a unique "context" in the log object to make the source more easily
identifiable:

```tsx
import { useEffect } from 'react';
import type { FC } from 'react';
import { logger } from 'tachyon-logger';
import { errorMessageFromCatch } from 'tachyon-utils';

const SomeComponent: FC = () => {
  useEffect(() => {
    try {
      const res = fetch(...);
    } catch (e) {
      logger.error({
        context: 'SomeComponent',
        message: errorMessageFromCatch(e),
      });
    }
  });
}
```

## Handling Dates

Tachyon apps are server-side rendered which can cause surprising behavior when a
user who is in a different time-zone than the server accesses a page. When
possible, use phrasing like `3 hours ago` to communicate time rather than
`at 2pm today`. This will allow information to be cached, making the app more
performant for your users and preventing mix-ups from time-zones.

Of note, even things like `last Tuesday` are problematic due to time-zones and
should be avoided when possible.

Sometimes, avoiding specific times or dates is impossible. When this happens,
Tachyon provides a mechanism to force content to be rendered on the client's
device, at the cost of speed and caching. See below for an example.

```tsx
import type { FC } from 'react';
import { ClientOnly } from 'components/common/ClientOnly';
import {
  SomeTimeComponent,
  SomeTimeComponentProps,
} from '../some-time-component';

export const ClientTimeComponent: FC<SomeTimeComponentProps> = ({
  start,
  end,
}) => (
  <ClientOnly>
    <SomeTimeComponent start={start} end={end} />
  </ClientOnly>
);
```

### Warning about page "bouncing" from ClientOnly usage

Simply forcing content to render on the client often isn't enough to provide a
good experience. When content is loaded after initial content, you will need to
ensure that there isn't page "bouncing" by having content appear which moves
content after it. This means that any content you put inside `<ClientOnly>`
should leave space for itself. This might mean putting your `<ClientOnly>`
content inside a component that has a height that does not change regardless of
the content in order to prevent the content below it from changing place when
your date arrives in the space where is is required.

## Useful Feature Utilities From Tachyon-Utils

`displayName`: Renders a user's display name with international formatting as
necessary `formatVideoLength`: Outputs a human readable video length
`impressionListener`: Used to detect if an element is partially or fully visible
to the user. `estimateImageCountForViewport`: Determine how many images to
render for a given space.
