# Naming Conventions in the Tachyon Repo

## Filesystem

We use a very similar approach to naming files and directories as you'll find in
the [React](https://github.com/facebook/react) and
[Relay](https://github.com/facebook/relay) repositories, with names designed to
give context for someone browsing the file tree so that they can immediately
recognize what they are looking at.

The specific file naming conventions also cause various lint rules to be enabled
for strictness or disabled for convenience so it is important to stick to the
conventions.

## React component files

We follow the pattern of having the filename mirror the "default" export of the
file. While we don't actually use default exports, our files mirror the same
single-responsibility principle with one main thing per file. Thus a file
exporting `<Foo>` would be named `Foo.tsx`. Once that file gains tests, we
switch to `Foo/index.tsx` for the component (adding a new enclosing directory of
the component name) and `Foo/test.tsx` for its tests in order to limit visual
clutter.

## Function files

Similar to component files, the filename should mirror the "default" function,
again keeping to a single function per file. Thus a file exporting function
`bar()` would be named `bar.ts`. Once that file gains tests, we prefer the same
pattern of `bar/index.ts` and `bar/test.ts`

## Test Helper Files

These files should be named `foo/test-mocks.ts` or `foo/test-utils.ts`. They
should not be re-exported by their corresponding `index.ts` file to avoid
accidental inclusion of test dependencies that might not support tree shaking
and must be imported directly (`import { SomeUtil } from './foo/test-utils';`).

Using these specific file names also allow for other conveniences:

- Ability to use `any` and non-null assertions (`!`)
- Ability to do file level code coverage disabling through
  `// istanbul ignore file: mock / util`

## Modules

For directories that contain several components or functions, we prefer using
kebab-case (similar to package names themselves). Thus a directory that holds
utils related to "routing" could be called `routing-utils`, and could contain
several function or component files/directories.

_Note: We are in the midst of transitioning to this module naming pattern, so
you may not see it consistently everywhere yet. Please name all new module
directories in this manner, and feel free to convert any others that you come
across if you are so motivated._

## Markdown Files

Markdown files must be named in kebab-case for them to be properly handled by
our doc ingestion process in the Sky Map app. For example, this doc is named
`naming-conventions.md`.
