# React Components in the Tachyon Repo

## Organization

React Components must be organized using our Tachyon
[naming conventions](./naming-conventions.md):

- One folder per component, named to match the component
- The component file must be named index.tsx
- The component test file lives with the implementation file and must be named
  test.ts(x) which is required for Jest to identify and run the file.
- Sub-components may live in the component folder, following the same
  conventions

Example:

```
./SomeComponent/
  index.tsx
  test.ts
  SomeComponentSubComponent/
    index.tsx
    test.ts
```

## Component File Conventions

- Prefer Function Components + Hooks
- Name the prop type to match the component with the "Props" suffix
- Break out sufficiently complex hooks into custom Hooks and test separate from
  the component

```tsx
import type { FC } from 'react';

export type SomeComponentProps = {
  foo: string;
  onClick: () => void;
};

export const SomeComponent: FC<SomeComponentProps> = ({ foo, onClick }) => {
  /*...*/
};
```

If the component is to be wrapped by one or more HoCs, suffix the component
implementation with "Base":

```tsx
import type { FC } from 'react';
import { memo } from 'react';

export const SomeComponentBase: FC = () => {
  /*...*/
};

export const SomeComponent = memo(SomeComponentBase);
```

## Testing React Components

We use [tachyon-test-utils](../packages/tachyon-core/test-utils/README.md) to
simplify [Enzyme](https://enzymejs.github.io/enzyme/) component test scaffolding
through `createShallowWrapperFactory` and `createMountWrapperFactory`.

```ts
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { SomeComponent } from '.';

describe(SomeComponent, () => {
  const setup = createShallowWrapperFactory(SomeComponent, () => ({
    foo: 'some-string',
    onClick: jest.fn(),
  }));

  it('renders a button', () => {
    const { props, wrapper } = setup();
    expect(wrapper.find('button')).toExist();
  });
});
```

See [tachyon-test-utils](../packages/tachyon-core/test-utils/README.md) for more
testing examples.

_Tip: Through `jest-enzyme` many
[Enzyme specific assertions](https://github.com/FormidableLabs/enzyme-matchers#assertions)
are available as [Jest matchers](https://jestjs.io/docs/en/using-matchers)._

### Shallow Testing Item Renderer-type Components

Some components such as those that manage virtualization use an
`items`/`itemRenderer` prop pattern or similar:

```tsx
<VirtualizedList
  items={[{ name: 'Derek' }, { name: 'Risto' }]}
  itemRenderer={(item) => (
    <div>
      <h1>{item.name}</h1>
    </div>
  )}
/>
```

Rather than mount testing this pattern, which can add a lot of additional mock /
test complexity, use Jest's built in helpers for finding an object with a
specific field in an array:

```tsx
it('only provides Derek to VirtualizedList when allowed', () => {
  const { wrapper } = setup({ allowDerek: true });

  expect(wrapper.find(ItemRenderer).prop('items')).toContainEqual(
    expect.objectContaining({ name: 'Derek' }),
  );
});
```

### Debugging Component Tests

`wrapper.debug()` will output the current JSX for a test case in the console:

```tsx
it('passes the HOC prop to the wrapped consumer', () => {
  const { wrapper } = setup();
  console.log(wrapper.debug());
});
```
