# Creating React Contexts

This guide defines the file structure, naming conventions, and testing patterns
used when creating Contexts in Tachyon. Before proceeding ensure that you've
read React's [Context guide](https://reactjs.org/docs/context.html).

## Defining A React Context

Create a new folder / file:
`<some-project>/src/components/.../ActionRoot/index.tsx` with the following:

```tsx
import { FC, createContext, useContext, useMemo, useState } from 'react';

type Action = 'init' | 'validate' | 'complete';

// Name the context's type using CamelCase
type ActionContext = {
  currentAction: Action;
  onPhaseComplete: () => void;
};

// 1) Name the context using snakeCase
// 2) Define a sane context default
// 3) Ideally we don't export to this allow easy evolution in the future
const actionContext = createContext<ActionContext>({
  currentAction: 'init',
  onPhaseComplete: () => undefined,
});

// Export a hook to allow the context to be consumed
// istanbul ignore next: omit from test coverage since it is trivial
export function useAction(): ActionContext {
  return useContext(actionContext);
}

type ActionRootProps = {
  defaultAction: Action;
};

export const ActionRoot: FC<ActionRootProps> = ({
  children,
  defaultAction,
}) => {
  // Caution! Frequent state updates will cause all children (even non-Context consumers) to
  // re-render just like any other React component. Place this component in the component hierarchy accordingly.
  // If you plan to update a Root's state value frequently (every 0-5 seconds), and the context sits above a complex
  // and expensive to re-render component hierarchy, consider an alternative to maintain smooth run-time performance.
  // Feel free to reach out to EMP for suggestions.
  const [currentAction, setCurrentAction] = useState(defaultAction);

  // Memoize our context so that if this component is re-rendered, it doesn't
  // cause a render cascade to all consumers (lint rules will help enforce this)
  const ctx = useMemo(() => {
    return {
      currentAction,
      onPhaseComplete: () => {
        setCurrentAction((action) => {
          if (action === 'init') {
            return 'validate';
          }

          if (action === 'validate') {
            return 'complete';
          }
        });
      },
    };
  }, [currentAction]);

  return <actionContext.Provider children={children} value={ctx} />;
};
```

## Testing

This assumes you are familiar with the basics of React Component
[testing](./react-components.md#testing-react-components).

### Testing React Context Providers

The simplest way to test React Context Providers is to create a dummy consumer
that reads from the Context's Provider and makes asserting and consuming the
context easy. Using the example above:

```tsx
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { ActionContext, actionContext, ActionRoot } from '.';

const CtxReader: FC<SomeContext> = () => null;

describe(ActionRoot, () => {
  const setup = createMountWrapperFactory(ActionRoot, () => ({
    children: (
      <actionContext.Consumer children={(ctx) => <CtxReader {...ctx} />} />
    ),
  }));

  it('sets the expected value on the context', () => {
    const { wrapper } = setup();
    expect(wrapper.find(CtxReader)).toHaveProp({ currentAction: 'init' });

    act(() => {
      wrapper.find(CtxReader).prop('onPhaseComplete')!();
    });
    expect(wrapper.find(CtxReader)).toHaveProp({ currentAction: 'validate' });
  });
});
```

### Testing A Hook That Consumes Context

Refer to the
[custom hooks](./custom-hooks.md#testing-custom-hooks-with-mock-contexts)
documentation.

### Testing A Component That Consumes Context

The `createMountWrapperFactory` utility provided by `tachyon-test-utils`
includes
[Context mocking helpers](../packages/tachyon-core/test-utils/README.md#mocking-react-context-mount-only-currently)
to simplify unit testing of components that consume contexts.
