# Styling Components in the Tachyon Repo

The tachyon monorepo uses [styled-components](https://www.styled-components.com)
as its solution for creating custom styles for apps, packages, examples, and
components. It is a CSS-in-JS solution with a large ecosystem that enables
features such as easy critical-path style inclusion for SSR applications and
fully testable single-file components.

## Basic Use

Since styles are defined in tagged template literals, you can use normal JS
string operations to insert constants from JS, but otherwise you just write
vanilla CSS.We use `stylelint` across the repo as well to ensure validity and
consistency.

### Conventions

As `styled-components` creates React elements for use in JSX trees, we follow
the convention of prefixing all wrapped components with `Sc`, as you'll see in
the following examples.

This also means that CSS class names are unnecessary and should be avoided.

### Basic Styling

The simplest usage is attaching styles to primitive HTML tags.

```tsx
import type { FC } from 'react';
import styled from 'styled-components';

const ScContainer = styled.div`
  width: 60%;
`;

const SomeFeature: FC = () => <ScContainer>{/* ... */}</ScContainer>;
```

### Consuming Constants

The simplest usage is attaching styles to primitive HTML tags.

```tsx
import styled from 'styled-components';
import { BG_COLOR } from './some/config';

const ScCustomBgContainer = styled.div`
  background: ${BG_COLOR};
`;
```

### Consuming Core-UI Tokens

```tsx
import styled from 'styled-components';
import { StaticTokenMap } from 'twitch-core-ui-tokens';

const ScBrandedContainer = styled.div`
  background-color: ${StaticTokenMap['color-twitch-purple']};
`;
```

### Dynamic Styles

Instead of toggling class names, `styled-components` allows you to pass props in
order to dynamically generate the correct styling. You have to define the
additional props the component will receive, and those props will be merged with
the props of the primitive or wrapped component.

Props that are used for CSS styling only, and should not be passed down to the
augmented component itself, should be marked as
[transient props](https://styled-components.com/docs/api#transient-props) using
the `$` prefix.

```tsx
import type { FC } from 'react';
import styled from 'styled-components';
import { Button, ButtonType } from 'core-ui';

type ScStatusBoxProps = {
  $success: boolean;
};

const ScContainer = styled.div<ScStatusBoxProps>`
  color: ${({ $success }) => ($success ? 'green' : 'red')};
`;

type ScButtonProps = {
  $height: number;
};

const ScButton = styled(Button)<ScButtonProps>`
  height: {({ $heightRem }) => $heightRem}rem;
`;

type StatusBoxProps = {
  heightRem: number;
  success: boolean;
};

const StatusBox: FC<StatusBoxProps> = ({ children, heightRem, success }) => (
  <ScContainer $success={success}>
    <ScButton fullWidth $heightRem={heightRem} type={ButtonType.Secondary}>
      {children}
    </ScButton>
  </ScContainer>
);
```

### Utilities / "Mixins"

Since `styled-components` relies on tagged template literals, any Javascript
functions can be used inside of them to generate output. This means you can
extract logic into reusable chunks.

```tsx
import type { CSSObject } from 'styled-components';

type SquareSizeProps = {
  sizeRem: number;
};

/**
 * Generates equal height and width css properties. Returns empty object
 * in the case where you somehow defeat TypeScript.
 */
export function squareSize({ sizeRem }: SquareSizeProps): CSSObject {
  const size = `${sizeRem}rem`;
  return { height: size, width: size };
}

const ScSquareContainer = styled.div`
  ${squareSize({ sizeRem: 30 })}
`;
```

The EP team maintains a library of
[specific helpers inside `tachyon-more-ui`](https://git.xarth.tv/pages/emerging-platforms/tachyon/d/packages/tachyon-core/more-ui/#mixins-for-styled-components).
If you discover a pattern that is frequently re-used, you're encouraged to
extract it either into the local feature/app context or into `tachyon-more-ui`
itself if the pattern is sufficiently general.

In addition, the [polished library](https://polished.js.org/docs/) is also
available, providing a large collection of standardized helpers for working with
common CSS patterns (like `clearFix`), color utilities (like `transparentize`
which can add/change alpha to color strings), CSS shorthands (like sizing, etc),
and more. Note that there are sometimes stricter versions of `polished`
utilities in `tachyon-more-ui` in order to expose things like `sizeRem: number`
instead of `size: string`, and in those cases the `tachyon-more-ui` utilities
should be preferred.

In general, such function usage is encouraged because it frequently leads to
more readable style definitions and also helps reduce bundle-size.

### Extensibility

You can extend existing components to give them extra styles without redefining
a style block entirely.

```tsx
import styled from 'styled-components';

const ScBodyText = styled.p`
  font-family: sans-serif;
`;

const ScCenteredText = styled(ScBodyText)`
  text-align: centered;
`;
```

_`<ScCenteredText />` will have both properties applied._

#### Enhancing Core-UI Components

You can also extend more complex components, especially useful when interacting
with Core UI. You should not use this approach to get around Core UI's built in
capabilities or opinions, but rather to provide properties that Core UI does not
handle like height.

```tsx
import styled from 'styled-components';
import { Layout } from 'twitch-core-ui';

const ScInfoBox = styled(Layout)`
  height: 4rem;
  width: 50%;
`;
```

### Defining Global Styles

Use
[createGlobalStyle](https://styled-components.com/docs/api#createglobalstyle) to
apply global styling.

```tsx
import { createGlobalStyle } from 'styled-components';

const ScBody = createGlobalStyle`
  body {
    height: 100vh;
    width: 100vw;
  }
`;

const App = () => {
  return (
    <>
      <ScBody />
      {/* ... */}
    </>
  );
};
```

### Advanced use cases

The [styled-components docs](https://www.styled-components.com/docs) are very
helpful and cover more advanced usages with concise explanations and examples.

## Testing

The
[jest-styled-components](https://github.com/styled-components/jest-styled-components)
package allows for testing of the generated styles. It adds the
`.toHaveStyleRule()` matcher to Jest, allowing assertions on styled-components.
It is worth reading the
[toHaveStyleRule()](https://github.com/styled-components/jest-styled-components#tohavestylerule)
documentation for advanced examples on selecting and simulating things like
media or element states (:focus, :hover, etc).

The following component can be tested in both shallow and mount testing:

```tsx
import styled from 'styled-components';

export const ScExample = styled.div<{ color: string }>`
  color: ${({ color }) => color};
`;
```

### Direct Shallow Testing

```tsx
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { ScExample } from '.';

const setup = createShallowWrapperFactory(ScExample, () => ({
  color: 'red',
}));

it('has the correct styles', () => {
  const { wrapper } = setup();
  expect(wrapper).toHaveStyleRule('color', 'red');

  wrapper.setProps({ color: 'green' });
  expect(wrapper).toHaveStyleRule('color', 'green');
});
```

### Mount Testing

Use [enzymeFind()](https://styled-components.com/docs/api#enzymefind) to find
the Styled-Component in an Enzyme mount test.

```tsx
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { enzymeFind } from 'styled-components/test-utils';

const setup = createShallowWrapperFactory(ScExample, () => ({
  color: 'red',
}));

it('has the correct styles', () => {
  const { wrapper } = setup();
  expect(enzymeFind(wrapper, ScExample)).toHaveStyleRule('color', 'red');

  wrapper.setProps({ color: 'green' });
  expect(enzymeFind(wrapper, ScExample)).toHaveStyleRule('color', 'green');
});
```

### Validating Advanced CSS Selectors

Some Styled-Components contain style rules that are applied only for specific
states including:

- media query selectors
- (`:hover`, `:focus`, etc)
- descendent elements (`& > div`)

For these cases, you'll want to use the optional third argument object of the
[toHaveStyleRule](https://github.com/styled-components/jest-styled-components#tohavestylerule)
matcher.

As an example of asserting a configurable hover state color for a Styled
Component:

```tsx
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import styled from 'styled-components';

const ScContainer = styled.div<{ color: string; hoverColor: string }>`
  color: ${({ color }) => color};

  & :hover {
    color: ${({ hoverColor }) => hoverColor};
  }
`;

const setup = createShallowWrapperFactory(ScContainer, () => ({
  color: 'red',
  hoverColor: 'purple',
}));

it('has the correct styles', () => {
  const { wrapper } = setup();
  expect(wrapper).toHaveStyleRule('hoverColor', 'purple', {
    modifier: '& :hover',
  });

  wrapper.setProps({ hoverColor: 'yellow' });
  expect(wrapper).toHaveStyleRule('color', 'yellow', {
    modifier: '& :hover',
  });
});
```

#### Debugging Styled Components In Tests

Use `expect(wrapper).toMatchInlineSnapshot();` to print out a human readable
version of the current Styled-Component(s) being rendered.

## SSR Usage

Covered in the
[styled-compontents docs](https://www.styled-components.com/docs/advanced#server-side-rendering),
but the summary is that the app is wrapped in a provider that captures all the
necessary styles during a server-side render pass. After the app content has
been determined, both it and the resulting "critical path" styles are rendered
into the HTML for being sent to the client. This ensures that the HTML doc is as
small and efficient as possible for the first client render.
