# Unit Testing in Tachyon

## Basics

Tachyon uses [Jest](https://jestjs.io/docs/en/getting-started) as its testing
framework. This guide will cover general unit testing patterns. For tips on
testing specific patterns (React Components / Hooks, Styled Component, etc),
refer to their guides.

Before getting started, read our
[test coverage policy](../policies/test-coverage.md). Refer to
[CodeCov](https://codecov.xarth.tv/ghe/emerging-platforms/tachyon) for the
current state and distribution of Tachyon unit test coverage.

## Unit Testing Style

Use the general test writing style as defined at
[Better Specs](http://www.betterspecs.org/).

Tests should be written using the BDD syntax that Jest provides. In a root level
`describe`, specify the component or function you intend to test directly:

```ts
describe(SomeComponent, () => {
  it('computes an expected value given a valid input', () => {
    expect(...).toEqual(true);
  });
});
```

All tests should be contained within a `describe` block, and all setup code
(other than mocking) relevant to the tests should be contained within this
`describe` block. Similar to trailing commas, this makes it easy for future devs
to expand the test file without confusion.

## Scaffolding Test Data

Tachyon provides [Faker](https://github.com/marak/Faker.js/#api) in a test
context that contains a ton of useful data generation utilities.

**DO NOT attempt to use Faker in production code. It is massive.**

## Mocking Imports And Modules

Jest provides the ability to
[mock](https://jestjs.io/docs/en/mock-functions#mocking-modules) imports and
modules.

### Managing Mock Lifecycles

Jest is configured to
[clear all mocks](https://jestjs.io/docs/mock-function-api#mockfnmockclear)
between every test. This means that all call history is cleared automatically,
but implementations are left in place. This prevents calls from leaking across
tests, and frees you from the related boilerplate (in fact there is a lint rule
to discourage this). If you need to reset implementations between tests, using
`[mock].mockReset()` or `jest.resetAllMocks()` is still necessary.

### Mocking Local Dependencies

Mocking a file in the same app or package is straight forward:

```ts
import { someUtil } from './utils';

jest.mock('./utils');

const mockSomeUtil = someUtil as jest.Mock;
```

### Mocking Dependencies From Packages

To safely mock one or more dependencies from a package without entirely
clobbering all of its exports, use
[requireActual](https://jestjs.io/docs/en/jest-object#jestrequireactualmodulename):

```ts
import { doSomething } from 'some-package';

// Mocks only the dependency you wish to control but preserves the rest
jest.mock('some-package', () => ({
  ...jest.requireActual('some-package'),
  doSomething: jest.fn(),
}));

const mockDoSomething = doSomething as jest.Mock;
```

## Mocking Fetch Calls

Since we use GraphQL for most of our data, it should be rare that you need to
use a custom data fetch. GraphQL data testing is handled without fetch mocking.

We use [jest-fetch-mock](https://github.com/jefflau/jest-fetch-mock) to handle
mocking any fetch calls (with appropriately updated types). It doesn't
differentiate mocks between different URLs like some libraries and is meant to
be used on a controlled per-test basis if you need to interact with either the
request or response side of the fetch (as opposed to just ensuring that outbound
calls are not made during tests). You'll want to ensure that you reset the fetch
mocks between tests.

To ensure that fetch was called with the right arguments, you just treat fetch
like a normal jest mock function (`.toHaveBeenCalledWith()`,
`.toHaveBeenCalledOnce()`, etc):

```ts
import { somethingThatInvokesAFetch } from '.';

describe(somethingThatInvokesAFetch, () => {
  beforeEach(() => {
    fetch.resetMocks();
    // in this test we don't care about the return value
    fetch.mockResponseOnce('');
  });

  it('calls fetch correctly', () => {
    somethingThatInvokesAFetch();

    expect(fetch).toHaveBeenCalledWith(
      'target_url',
      // ... various fetch options
    );
  });
});
```

Note that it can be useful to use `expect.objectContaining()` to omit any parts
of your fetchOptions that can vary across tests:

```ts
const [[url, fetchOpts]] = fetch.mock.calls;
expect(fetchOpts).toEqual(
  expect.objectContaining({
    // ... various stable attributes
  });
);
// ensure the presence of the attributes omitted from above
expect(fetchOpts.time).toBeDefined();
```

To test how a method would've handled the return of a fetch, you need to supply
a mock return value:

```ts
import { someDataHandlingMethod } from '.';

describe(someDataHandlingMethod, () => {
  beforeEach(() => {
    fetch.resetMocks();
  });

  it('handles fetched data correctly', () => {
    // use JSON.stringify if you're expecting JSON, otherwise use a plain string
    // for text responses.
    fetch.mockResponseOnce(
      JSON.stringify({
        data: {
          /// ...etc
        },
      }),
    );

    expect(someDataHandlingMethod()).toReturn(correctlyParsedData);
  });
});
```

[jest-fetch-mock](https://github.com/jefflau/jest-fetch-mock) has more examples
as well, but you should avoid the way those examples do `fetch.mock.calls[0][1]`
in favor of `expect(fetch).toHaveBeenCalledWith(...)` for better readability and
to ensure that you are testing the entire signature.
