import {Tracker, COOKIE_NAME} from "src/tracker";
import ILogger from "src/logger/ilogger";
import find from "lodash.find";

describe("tracker", () => {
  describe("parameters passed to logEvent", () => {
    const APP_VERSION = "version1.2.3";
    const CHANNEL = "monsterdog";
    const CONTENT_TYPE = "content";
    const PREFERRED_LANGUAGE = "blah";
    const RECEIVED_LANGUAGE = "blarg";
    let tracker, logger;
    beforeEach(() => {
      logger = jasmine.createSpyObj("MockLogger", ["logEvent"]);
      document.cookie =
        `${COOKIE_NAME}=; expires=Thu, 01 Jan 1970 00:00:01 GMT;`;
      tracker = new Tracker(
        logger,
        APP_VERSION,
        CHANNEL,
        CONTENT_TYPE,
        PREFERRED_LANGUAGE,
        RECEIVED_LANGUAGE
      );
    })

    it("passes the event", () => {
      tracker.trackEvent("event", {});
      var args = logger.logEvent.calls.argsFor(0);
      expect(args[0]).toEqual("event");
    });

    it("passes the parameter object correctly", () => {
      let subject;
      tracker.trackEvent("event", {});
      subject = logger.logEvent.calls.argsFor(0)[1];

      expect(subject.app_version).toEqual(APP_VERSION);
      expect(subject.channel).toEqual(CHANNEL);
      expect(subject.content_type).toEqual(CONTENT_TYPE);
      expect(subject.preferred_language).toEqual(PREFERRED_LANGUAGE);
      expect(subject.received_language).toEqual(RECEIVED_LANGUAGE);
    });

    it("builds parameters from the environment correctly", () => {
      let subject;
      tracker.trackEvent("event", {});
      subject = logger.logEvent.calls.argsFor(0)[1];

      expect(subject.browser).toEqual(navigator.userAgent);
      expect(subject.domain).toEqual(window.document.domain);
      expect(subject.url).toEqual(window.document.location.href);
      expect(subject.referrer_host).toEqual("localhost:9876");
      expect(subject.referrer_url).toEqual(window.document.referrer);
    });

    describe("device ID", () => {
      it("generates a 32 character string", () => {
        tracker.trackEvent("event", {});
        expect(logger.logEvent.calls.argsFor(0)[1].device_id.length)
          .toEqual(32);
      });

      it("provides the same ID with each event", () => {
        tracker.trackEvent("event", {});
        let deviceIdTheFirst = logger.logEvent.calls.argsFor(0)[1].device_id;
        tracker.trackEvent("event", {});
        let deviceIdTheSecond = logger.logEvent.calls.argsFor(0)[1].device_id;
        expect(deviceIdTheFirst).toEqual(deviceIdTheSecond);
      });

      it("stores the device ID in a cookie", () => {
        tracker.trackEvent("event", {});
        let deviceId = logger.logEvent.calls.argsFor(0)[1].device_id;
        let cookieValue = find(document.cookie.split(";"), (cookie) => {
          return cookie.substr(0, COOKIE_NAME.length) === COOKIE_NAME;
        }).split("=")[1];
        expect(deviceId).toEqual(cookieValue);
      });

      it("uses the cookie value if it is present", () => {
        let cookieValue = find(document.cookie.split(";"), (cookie) => {
          return cookie.substr(0, COOKIE_NAME.length) === COOKIE_NAME;
        }).split("=")[1];
        let otherLogger = jasmine.createSpyObj("MockLogger", ["logEvent"]);
        new Tracker(
          otherLogger,
          APP_VERSION,
          CHANNEL,
          CONTENT_TYPE,
          PREFERRED_LANGUAGE,
          RECEIVED_LANGUAGE
        ).trackEvent("event", {});
        let deviceId = otherLogger.logEvent.calls.argsFor(0)[1].device_id;
        expect(cookieValue).toEqual(deviceId);
      });
    })
  });
});
