import ILogger from "./logger/ilogger";
import assign from "lodash.assign";
import find from "lodash.find";

export const COOKIE_NAME = "unique_id";
const SECONDS_IN_YEAR = 60 * 60 * 24 * 365;
const MAX_AGE_IN_SECONDS = 10 * SECONDS_IN_YEAR;

export class Tracker {
  private trackingParams: Object;
  private deviceId: string;

  constructor(
    private logger,
    appVersion: string,
    channel: string,
    contentType: string,
    preferredLanguage: string,
    receivedLanguage: string
  ) {
    this.trackingParams = {
      app_version: appVersion,
      channel: channel,
      content_type: contentType,
      device_id: this.getOrCreateDeviceId(),
      browser: navigator.userAgent,
      domain: window.document.domain,
      url: window.document.location.href,
      referrer_host:
        (window.document.referrer.match("https?://([^/]+)") || [])[1],
      referrer_url: window.document.referrer,
      preferred_language: preferredLanguage,
      received_language: receivedLanguage
    };
  }

  trackEvent(event: string, additionalProperties: Object) {
    let properties = assign({}, additionalProperties, this.trackingParams);
    this.logger.logEvent(event, properties);
  }

  private getDeviceIdFromCookie(): string {
    let cookies = document.cookie.split(";");
    let qualifier = `${COOKIE_NAME}=`;
    let cookie = find(cookies,
      (cookie) => {
        return cookie.substr(0, qualifier.length) === qualifier;
      }
    );

    if (cookie) {
      return cookie.split("=").splice(1).join("=");
    }
  }

  private createAndSetDeviceId(length = 32): string {
    let possible = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";

    let text = "";
    for (let i = 0; i < length; i++) {
        let index = Math.floor(Math.random() * possible.length);
        text += possible.charAt(index);
    }

    this.deviceId = text;
    document.cookie = `${COOKIE_NAME}=${text};max-age=${MAX_AGE_IN_SECONDS};path=/`;

    return text;
  }

  private getOrCreateDeviceId(): string {
    return this.deviceId || this.getDeviceIdFromCookie() || this.createAndSetDeviceId();
  }
}