#!/usr/bin/env python3

import socket
import json
import argparse
import logging
import logging.handlers
import boto3
from time import sleep

import requests
import etcd


parser = argparse.ArgumentParser(description="ETCD Inserter")
parser.add_argument("--no-loop", required=False, default=False, dest="no_loop", help="don't loop, run only once", action="store_true")
parser.add_argument("--sleep", required=False, default=5, dest="sleep", help="how long (in seconds) to wait between loops", type=int)
parser.add_argument("--clusters-only", required=False, default=False, dest="clusters_only", help="dont copy services from pharah, simply copy clusters", action="store_true")
parser.add_argument("--services-only", required=False, default=False, dest="services_only", help="dont copy clusters from fs, simply copy services from pharah", action="store_true")
parser.add_argument("--env", required=False, dest="env", help="custom env to run against")
parser.add_argument("--files", required=False, dest="files", nargs="+", help="which cluster files to copy")

logging.basicConfig(level=logging.DEBUG, format='etcd_inserter %(levelname)s [%(threadName)s:%(module)s.py:%(lineno)d] %(message)s')
logger = logging.getLogger('etcd_inserter')

SALT_BASE_PATH = '/srv/salt/active/src/admin/files'

ENV_TO_URL = {
    'bebo-prod': 'http://usw2-pharah.aws.bebo.com/service',
    'dev': 'http://usw1-pharah.aws.bebo-dev.com/service',
    'local': 'http://localhost:8650/service'
}

def get_api_key(env):
    if env == "bebo-prod":
        REGION = "us-west-2"
    else:
        REGION = "us-west-1"

    SSM = boto3.client("ssm", region_name=REGION)
    name = "/%s/etcd_inserter/API_KEY" % env
    res = SSM.get_parameter(
        Name=name,
        WithDecryption=True
    )
    return json.loads(res['Parameter']['Value'])

class EtcD(object):
    self_client = etcd.Client(host="localhost", port=2379)

    @classmethod
    def update_service(cls, service):
        key = service['name']
        return cls.set('service/{}'.format(key), json.dumps(service))

    @classmethod
    def set(cls, key, value):
        cls.self_client.write(key, value)

    @classmethod
    def delete(cls, key):
        cls.self_client.delete(key)

    @classmethod
    def list(cls, key):
        return [
            child['key'].split('/{}/'.format(key))[1]
            for child in
            cls.self_client.get(key)._children
        ]

class Inserter():
    def __init__(self, args):
        self.sleep = args.sleep
        self.loop = not args.no_loop

        self.services_only = args.services_only
        self.clusters_only = args.clusters_only

        self.env = args.env or self.get_env_from_fqdn()
        self.vpc_type = self.get_vpc_type()

        self.cluster_file_names = args.files or [
            '{}/etcd_{}_base.json'.format(SALT_BASE_PATH, self.vpc_type),
            '{}/etcd_{}_{}.json'.format(SALT_BASE_PATH, self.vpc_type, self.env)
        ]

    def get_env_from_fqdn(self):
        fqdn = socket.getfqdn()
        if 'dev' in fqdn:
            return 'dev'
        if 'ptr' in fqdn:
            return 'ptr'

        return 'bebo-prod'

    def get_vpc_type(self):
        my_mac = requests.get('http://169.254.169.254/latest/meta-data/network/interfaces/macs', timeout=0.5).text.split('/n')[0]
        vpc_id = requests.get('http://169.254.169.254/latest/meta-data/network/interfaces/macs/%s/vpc-id' % my_mac, timeout=0.5).text
        vpc_type = 'regional'
        if vpc_id == 'vpc-5d513838' or vpc_id == 'vpc-3ec5705a':
            vpc_type = 'master'

        return vpc_type

    def diff_for_delete(self, new, old):
        to_delete = []
        for elem in old:
            if elem not in new:
                to_delete.append(elem)

        return to_delete

    def start(self):
        self.run()
        while self.loop:
            try:
                self.run()
            except KeyboardInterrupt:
                self.loop = False
            except Exception as e:
                logger.exception('Uncaught exception in main run loop: %s' % e)

        sleep(self.sleep)

    def run(self):
        try:
            if not self.clusters_only:
                self.copy_services_from_pharah()
        except Exception:
            logger.exception('Error copying services from pharah')

        try:
            if not self.services_only:
                self.copy_clusters_to_etcd()
        except Exception:
            logger.exception('Error copying cluster to etcd')

    def copy_services_from_pharah(self):
        headers = {
            'X-Api-Key': get_api_key(self.env)
        }
        url = ENV_TO_URL[self.env]
        resp = requests.get(url, headers=headers)
        resp.raise_for_status()
        services = resp.json().get('result', [])
        for service in services:
            EtcD.update_service(service)

        new_service_names = [service['name'] for service in services]
        old_service_names = EtcD.list('service')
        to_delete_service_names = self.diff_for_delete(new_service_names, old_service_names)
        for service_name in to_delete_service_names:
            logger.info('Deleting {} service because it is not referenced in update'.format(service_name))
            EtcD.delete('service/{}'.format(service_name))

    def copy_clusters_to_etcd(self):

        logger.debug('file_names: %s', self.cluster_file_names)

        new_cluster_names = []

        for file_name in self.cluster_file_names:
            try:
                with open(file_name, "r") as json_file:
                    contents = json_file.read()
                    values = [v for v in contents.split("\n") if v]

                    for v in values:
                        try:
                            logger.info("Inserting: {}".format(v))
                            obj = json.loads(str(v))
                            key = obj["key"]
                            cluster_name = key.split('/')[1]
                            value = json.dumps(obj["value"])

                            EtcD.set(key, value)
                            new_cluster_names.append(cluster_name)
                        except Exception as e:
                            logger.exception("Failed to insert: {} from {} {}".format(v, file_name, e))
            except FileNotFoundError:
                logger.error('Unable to read JSON from %s', file_name)

        if not new_cluster_names:
            logger.info('Not deleting any clusters because no clusters read from fs')
            return

        old_cluster_names = EtcD.list('cluster')
        to_delete_cluster_names = self.diff_for_delete(new_cluster_names, old_cluster_names)
        for cluster_name in to_delete_cluster_names:
            logger.info('Deleting {} cluster because it is not referenced in update'.format(cluster_name))
            EtcD.delete('cluster/{}'.format(cluster_name))

if __name__ == '__main__':
    try:
        inserter = Inserter(parser.parse_args())
        inserter.start()
    except Exception as e:
        logger.exception(e)
