#!/usr/bin/env python3

import configparser
import socket
import subprocess
import logging

import requests
import boto3

logging.basicConfig(level=logging.INFO, format='bebo_route53 %(levelname)s [%(threadName)s:%(module)s.py:%(lineno)d] %(message)s')
logger = logging.getLogger(name="bebo_route53")

def get_meta_data_keys():
    return requests.get('http://169.254.169.254/latest/meta-data/').text.split('\n')

def resolve_dns_record(record_name):
    command = ['host', record_name]
    process = subprocess.Popen(command, stdout=subprocess.PIPE)
    out, err = process.communicate()
    if err is not None:
        logger.error('ERROR resolve_dns_record: {}'.format(err))
        return None

    out = out.decode('utf-8')
    if 'NXDOMAIN' in out:
        return None

    return list({line.split('has address ')[1] for line in filter(lambda line: 'has address' in line, out.split('\n'))})

class BeboRoute53(object):
    def __init__(self):
        self.region = requests.get("http://169.254.169.254/latest/meta-data/placement/availability-zone").text[:-1]
        self.client = boto3.client('route53', region_name=self.region)

        self.has_public_ip = False
        self.setup()

        self.run(self.private_domain, self.private_ip_address)

        if self.has_public_ip:
            self.run(self.public_domain, self.public_ip_address)

    def setup(self):
        meta_data_keys = get_meta_data_keys()
        self.has_public_ip = 'public-ipv4' in meta_data_keys

        self.full_hostname = socket.getfqdn()

        split_hostname = self.full_hostname.split('.')
        self.nodename = split_hostname[0]
        self.all_zones = self.client.list_hosted_zones_by_name()

        self.private_ip_address = requests.get('http://169.254.169.254/latest/meta-data/local-ipv4').text
        self.private_domain = '.'.join(split_hostname[1:])

        if self.has_public_ip:
            self.public_ip_address = requests.get('http://169.254.169.254/latest/meta-data/public-ipv4').text
            self.public_domain = '.'.join(split_hostname[2:])

    def run(self, domain, ip):
        record_name = self.nodename + "." + domain
        dns_addr = resolve_dns_record(record_name)
        logger.info("dns_addr for {}: {}, ip: {}".format(record_name, dns_addr, ip))

        if dns_addr is not None and ip in dns_addr:
            return

        zone_id = None

        for hosted_zone in self.all_zones['HostedZones']:
            if hosted_zone['Name'] == domain + '.':
                zone_id = hosted_zone['Id'].split('/')[-1]

        logger.info("adding: record_name: {}, domain: {}, ip: {}, zone_id: {}".format(record_name, domain, ip, zone_id))

        self.client.change_resource_record_sets(
            HostedZoneId=zone_id,
            ChangeBatch={
                'Comment': 'UPSERT {} ({}) to {} via bebo_route53.py'.format(ip, self.full_hostname, record_name),
                'Changes': [
                    {
                        'Action': 'UPSERT',
                        'ResourceRecordSet': {
                            'Name': record_name,
                            'Type': 'A',
                            'TTL': 60,
                            'ResourceRecords': [
                                {
                                    'Value': ip
                                },
                            ],
                        }
                    }
                ]
            }
        )

if __name__ == '__main__':
    try:
        BeboRoute53()
    except:
        logger.exception("failed to run bebo_route53.py")
        exit(1)
