#!/usr/bin/env python3

import socket
from multiprocessing import Process
import json
import logging
import logging.handlers
import subprocess
import argparse
import random
from time import sleep

import requests

import etcd

parser = argparse.ArgumentParser(description="Deploy Docker")
parser.add_argument("--cluster-name", dest="cluster_name", help="cluster to operate as")
parser.add_argument("--service-names", dest="service_names", help="service names to deploy", nargs="*")

logging.basicConfig(level=logging.DEBUG, format='docker_deploy %(levelname)s [%(threadName)s:%(module)s.py:%(lineno)d] %(message)s')
logger = logging.getLogger('docker_deploy')


SPEED = random.randint(10, 30)
DEFAULT_SERVICES = ['warg', 'wildling', 'rsyslog-forwarder']

'''
Service
{
    "name": "warg",
    "version": "0.0.1",
    "volumes": ['/data/dispatcher:/data/dispatcher'],
    "env": [],
    "health_port": 3000
}
'''

'''
Cluster
{
    "name": "admin",
    "service_names": ["etcd", "wildling", "stark"],
    "ignore": true,
    "security_group": {
        "name": "admin-v1.0.0",
        "rules": [{
            "type": "ingress",
            "from": 80,
            "protocol": "tcp",
            "desc": "http in",
            "to": "0.0.0.0/0",
            "to_ipv6": "::/0"
        }]
    },
    "autoscale": false,
    "http_proxy_port": 1000
}
'''

def get_etcd_port():
    return 2379

class EtcD(object):
    self_client = etcd.Client(host="localhost", port=get_etcd_port())

    @classmethod
    def update_cluster(cls, cluster_name, cluster):
        return cls.set("cluster/{}".format(cluster_name), json.dumps(cluster))

    @classmethod
    def get_health_status(cls):
        key = 'health/{}'.format(socket.getfqdn())
        return cls.get_json(key)

    @classmethod
    def update_service(cls, service):
        key = service['name']
        return cls.set('service/{}'.format(key), json.dumps(service))

    @classmethod
    def get_all_keys(cls, path=""):
        items = cls.get(path)._children
        keys = [item["key"] for item in items]
        dirs = [item for item in items if item["dir"]]
        return keys.extend([cls.get_all_keys(_dir["key"] for _dir in dirs)])

    @classmethod
    def set(cls, key, value):
        cls.self_client.write(key, value)

    @classmethod
    def get(cls, key):
        try:
            return cls.self_client.get(key)
        except etcd.EtcdKeyNotFound:
            return None

    @classmethod
    def get_json(cls, key):
        resp = cls.get(key)
        try:
            return json.loads(resp.value)
        except Exception:
            return {}

def get_running_status_for_service(service_name):
    cmd = ['/usr/bin/docker', 'inspect', service_name]
    try:
        proc = call(cmd)

        stdout = proc.stdout.decode('utf-8').strip()
        inspected = json.loads(stdout)[0] #inspect returns array of 1 object
        running_version = inspected['Config']['Image'].split(":")[1]
        return inspected['State']['Running'], running_version
    except (subprocess.CalledProcessError, KeyError):
        return None, None

def get_image_id_for_image(image):
    cmd = ['/usr/bin/docker', 'image', 'inspect', image, '--format="{{json .Id}}"']
    try:
        proc = call(cmd)
    except subprocess.CalledProcessError:
        return None
    stdout = proc.stdout.decode('utf-8')
    image_id = stdout.split(":")[1].split('"')[0]
    return image_id

def call(command):
    #  logger.info('Calling {}'.format(command))
    return subprocess.run(command, stdout=subprocess.PIPE, check=True)

def get_env(fqdn):
    if 'dev' in fqdn:
        return 'dev'
    if 'ptr' in fqdn:
        return 'ptr'

    return 'bebo-prod'

def get_vpc():
    fqdn = socket.getfqdn()
    try:
        return fqdn.split(".")[0].split("-")[0]
    except KeyError:
        return "unknown"

def download_file(url, destination):
    r = requests.get(url, stream=True)
    try:
        r.raise_for_status()
        with open(destination, 'wb') as f:
            for chunk in r.iter_content(chunk_size=1024):
                if chunk: # filter out keep-alive new chunks
                    f.write(chunk)
    except requests.HTTPError as e:
        raise e

class NotFoundException(Exception):
    pass

def cleanup_directory():
    docker_rmi_cmd = ['/usr/bin/docker', 'system', 'prune', '-af']
    call(docker_rmi_cmd)

def docker_list_running_containers():
    docker_ps_cmd = ['/usr/bin/docker', 'container', 'ls', '--format', "{{ .Names }}"]
    try:
        proc = call(docker_ps_cmd)
        stdout = proc.stdout.decode('utf-8').strip()
        return stdout.split('\n')
    except subprocess.CalledProcessError:
        return []

def docker_check_image(service):
    service_name = service['name']
    version = service['version']

    service_running, running_version = get_running_status_for_service(service_name)

    if service_running and running_version == version:
        #  logger.info('Not running remove for {} as the running_version matches version {}'.format(service_name, running_version))
        return False

    if not service_running:
        logger.debug('%s requires deploying because not running' % service_name)
    else:
        logger.debug('%s requires deploying because versions dont match (curr:%s, new:%s)' % (service_name, running_version, version))

    return True

def docker_pull(service):
    """
    Ensure the proper version of the service is loaded into docker
    """
    service_name = service['name']
    version = service['version']

    image = 'bebodev/' + service_name + ':' + version
    logger.info("Pulling {} ({}) from Docker registry".format(service_name, image))
    pull_cmd = ['/usr/bin/docker', 'pull', image]

    call(pull_cmd)

def docker_run(service):
    """
    Ensure the service is running, with the proper network, env, volumes, etc.
    """
    service_name = service['name']
    version = service['version']
    image = 'bebodev/' + service_name + ':' + version
    live_deploy = service.get('live_deploy', True)

    if not live_deploy:
        stoppable = EtcD.get_health_status().get('stoppable', False)
        service_running, _ = get_running_status_for_service(service_name)
        if service_running and not stoppable:
            return

    try:
        remove_cmd = ['/usr/bin/docker', 'rm', '-f', service_name]
        call(remove_cmd)
    except subprocess.CalledProcessError:
        pass

    fqdn = socket.getfqdn()
    env = get_env(fqdn)

    prefix = fqdn.split(".")[0]
    container_hostname = "{}.{}".format(prefix, service_name)

    run_cmd = [
        '/usr/bin/docker', 'run', '-d',
        '--network', 'host',
        '--hostname', container_hostname,
        '--name', service_name,
        '--log-driver', 'syslog',
        '--log-opt', 'syslog-format=rfc5424',
        '--log-opt', 'syslog-address=udp://localhost:514',
        '--log-opt', 'syslog-facility=local4',
        '--log-opt', 'tag={}'.format(service_name),
        #  '--user', '100051',
        '-e', 'BEBO_ENV={}'.format(env),
        '-v', '/data/dispatcher:/data/dispatcher'
    ]

    for env_addition in service['env']:
        run_cmd.extend(['-e', env_addition])

    for volume in service['volumes']:
        run_cmd.extend(['-v', volume])

    run_cmd.append(image)

    logger.info('Running up {} with {}'.format(service_name, run_cmd))

    call(run_cmd)

def pull_and_run(service):
    docker_pull(service)
    docker_run(service)

def get_service_names():
    args = parser.parse_args()

    cluster_name = args.cluster_name
    if not cluster_name:
        cluster_name = socket.getfqdn().split("-")[1]

    #  logger.debug('cluster_name: {}'.format(cluster_name))
    service_names = args.service_names
    cluster = {}
    if not service_names:
        try:
            cluster = EtcD.get_json('cluster/{}'.format(cluster_name))
            service_names = cluster.get('service_names', [])
            service_names.extend(DEFAULT_SERVICES)
        except NotFoundException:
            logger.warn('Services for "{}" not found in ETCD'.format(cluster_name))
        except Exception:
            logger.exception('Services for "{}" not found in ETCD'.format(cluster_name))

    if cluster and cluster.get('http_proxy_port', 0) != 0 and 'stencil' not in service_names:
        service_names.append('stencil')

    return service_names

def deploy_services():

    service_names = get_service_names()

    if not service_names:
        #  logger.debug("no services for cluster")
        return

    #  logger.info('service_names: {}'.format(service_names))
    deploy_procs = []
    for service_name in service_names:
        try:
            service = EtcD.get_json('service/{}'.format(service_name))
            #  logger.debug('Got {}: {}'.format(service_name, service))
            if service:
                do_deploy = docker_check_image(service)
                if do_deploy:
                    logger.info('Deploying {}'.format(service_name))
                    proc = Process(target=pull_and_run, args=(service,))
                    proc.start()
                    deploy_procs.append(proc)
            else:
                logger.info('{} Does not exist so not deploying'.format(service_name))

        except NotFoundException:
            logger.warn('ETCD deploy value for {} not found'.format(service_name))
        except Exception:
            logger.exception("couldn't deploy: {}".format(service_name))

    for proc in deploy_procs:
        proc.join()

    cleanup_directory()

def clean_up_extra_services():
    expected_service_names = get_service_names()
    non_default_service_names = [service_name for service_name in expected_service_names if service_name not in DEFAULT_SERVICES]
    logger.debug('expected_service_names: %s, non_default_service_names: %s' % (expected_service_names, non_default_service_names))

    if not non_default_service_names:
        return

    running_service_names = docker_list_running_containers()

    to_delete = [service_name for service_name in running_service_names if service_name not in expected_service_names]
    if not to_delete:
        return

    logger.info('deleting services: {}'.format(to_delete))

    delete_cmd = ['/usr/bin/docker', 'rm', '-f']
    delete_cmd.extend(to_delete)
    call(delete_cmd)

def main():
    run = True
    while run:
        try:
            deploy_services()
            clean_up_extra_services()
        except KeyboardInterrupt:
            run = False
        except Exception:
            logger.exception('Uncaught exception in main run loop')
        sleep(SPEED)

if __name__ == '__main__':
    try:
        main()
    except Exception as e:
        logger.exception(e)
