#!/usr/bin/env python3

import argparse
import json

from builtins import FileNotFoundError

import requests
import boto3

parser = argparse.ArgumentParser()
parser.add_argument('--path', required=True)
parser.add_argument('--name', dest='name', required=True)
parser.add_argument('--version', required=True)
parser.add_argument('--env', default='dev')
parser.add_argument('--live', action='store_true')

def get_api_key(env):
    if env == "bebo-prod":
        REGION = "us-west-2"
    else:
        REGION = "us-west-1"

    SSM = boto3.client("ssm", region_name=REGION)
    name = "/%s/etcd_inserter/API_KEY" % env
    res = SSM.get_parameter(
        Name=name,
        WithDecryption=True
    )
    return json.loads(res['Parameter']['Value'])


env_to_url = {
    'bebo-prod': 'https://pharah.bebo.com/service',
    'dev': 'http://usw1-pharah.aws.bebo-dev.com/service',
    'ptr': 'https://pharah.bebo-ptr.com/service',
    'greengoblin': 'https://greengoblin-lb.bebo-dev.com:8650/service'
}

def main():
    args = parser.parse_args()
    env = args.env

    if not env in env_to_url:
        print('ERROR: Unknown env: {)'.format(env))
        return

    try:
        with open(args.path) as etcd_file:
            contents = etcd_file.read().strip()
            body = json.loads(contents)
    except FileNotFoundError:
        body = {}

    body['version'] = args.version
    body['name'] = args.name
    body['live'] = args.live
    body['health_port'] = body.get('health_port', None)
    body['live_deploy'] = body.get('live_deploy', True)
    body['env'] = body.get('env', [])
    body['volumes'] = body.get('volumes', [])

    headers = {
        'X-Api-Key': get_api_key(args.env),
        'Content-Type': 'application/json'
    }

    url = env_to_url[args.env]

    print('Posting Service to {} : {}'.format(url, body))
    #  print('Headers: {}'.format(headers))

    response = requests.post(url, headers=headers, data=json.dumps(body))
    response.raise_for_status()
    print(response.status_code)

if __name__ == "__main__":
    main()
