const uuid = require('uuid');
const fs = require('fs');
const os = require('os');
const util = require('util');
const config = require('../Config');
const hostname = os.hostname();

var logger = null;

class Dispatcher {
  constructor(dirPath, queueName, pickup_rate_ms = 60 * 1000, pickup_delay_ms = 0, verbose) {
    this.queueName = queueName;
    this.pickup_rate_ms = pickup_rate_ms;
    this.pickup_delay_ms = pickup_delay_ms;
    this.dirPath = dirPath;
    this.verbose = verbose;
    this.deliveryTime = Math.floor(Date.now() + this.pickup_rate_ms + this.pickup_delay_ms, 10);
    if (logger == null) {
      logger = require('../Logger');
    }

    if (this.verbose) {
      logger.info(`BeboNodeCommons::Dispatcher:constructor constructed for ${queueName} queue`);
    }

    this._generateFileName = this._generateFileName.bind(this);
    this.writeEvent = this.writeEvent.bind(this);
    this.doWrite = this.doWrite.bind(this);
  }

  _generateFileName() {
    const deliveryTime = Math.floor(Date.now() + this.pickup_rate_ms + this.pickup_delay_ms, 10);
    if(Date.now() > this.deliveryTime){
      this.deliveryTime = deliveryTime;
    }
    const path = `${this.dirPath}/${this.queueName}__${this.deliveryTime}`;
    return path;
  }

  genIndex(when=Date.now()) {
    const { queueName } = this;
    const rn = new Date(when);
    return `${queueName}-${rn.getUTCFullYear()}-${rn.getUTCMonth()+1}-${rn.getUTCDate()}`;
  }

  writeEvent(data) {
    data._index = data._index || this.genIndex(data.event_dttm);
    data.env_tx = data.env_tx || config.env;
    data.hostname_tx = data.hostname_tx || hostname.split('.')[0];
    data.region_tx = data.region_tx || hostname.split('-')[0];
    if (!data.routing_key && data.category_tx && data.action_tx) {
      data.routing_key = `${data.category_tx}.${data.action_tx}`;
      if (data.label_tx) {
        data.routing_key += `.${data.label_tx}`;
      }
    }

    //really not a very nessecary step other than the fact that we use `writeEvent` everywhere so backwards compatible :)

    return this.doWrite(data).catch(err => {
      logger.error('BeboNodeCommons::Dispatcher:writeEvent uncaught error', err, err && err.stack ? err.stack : '', this.queueName);
      logger.debug(util.inspect(data));
    });
  }

  writeBuffer(buff) {
    return new Promise((res, rej) => {
      fs.writeFile(this._generateFileName(), buff, err => {
        if (err) return rej(err);
        return res();
      });
    });
  }

  doWrite(obj) {
    return new Promise((resolve, reject) => {
      if (!obj || typeof obj !== 'object') {
        return reject('no object passed into doWrite');
      }
      if (!obj._id) {
        obj._id = uuid.v4();
      }
      if (!obj.event_at) {
        obj.event_at = Date.now();
      }

      fs.appendFile(this._generateFileName(), JSON.stringify(obj) + '\n', 'utf8', err => {
        if (err) {
          return reject(err);
        }
        return resolve();
      });
    });
  }
}

module.exports = Dispatcher;
