const BANNED_SUFFIX = ['_json', '_tx', '_nr', '_ind', '_key', '_id', '_dec', '_sa'];
const DATE_SUFFIX = ['_dttm'];
const EPOCH_SUFFIX = ['_at'];
const DEFAULT_MAX_DEPTH = 10;

const os = require('os');
const hostname = os.hostname();

function isInt(n) {
  return Number(n) === n && n % 1 === 0;
}
function isFloat(n) {
  return Number(n) === n && n % 1 !== 0;
}

class Utils {

  static getDefaultESEventFromRequest(req) {
    var now = new Date();
    const defaults = {
      remote_ip_tx: req.ip,
      event_dttm: now.toISOString(),
      node_tx: hostname
    };
    if (req.socket) {
      defaults.remote_port_nr = req.socket.remotePort;
    }

    const ua = req.get('User-Agent');
    if (ua) {
      defaults.user_agent_tx = ua;
    }
    const xua = req.get('X-Riano-User-Agent');
    if (xua) {
      defaults.x_user_agent_tx = xua;
    }
    return defaults;
  }

  static makeUserESFriendly(user) {

    if (user == null) {
      return;
    }
    if (user.dataValues) {
      user = user.get();
    }
    let esUser = {
      user_id: user.user_id,
      username_tx: user.username,
      created_dttm: user.created_dttm,
    };
    if (user.campaign) {
      esUser.campaign = Utils.makeESFriendly(user.campaign, 0);
    }
    if (user.days != null) {
      esUser.days_cnt = user.days;
    }
    if (user.geo) {
      esUser.geo = user.geo;
    }

    for (let key of Object.keys(user)) {
      if (key.startsWith("has_")) {
        esUser[key + "_ind"] = user[key];
      }
    }

    if (user.deleted_dttm) {
      esUser.deleted_dttm = user.deleted_dttm;
    }
    return esUser;
  }

  static makeESFriendly(dict, MAX_DEPTH = DEFAULT_MAX_DEPTH, depth = 0) {

    if (dict == null) {
      return;
    }

    // FIXME make arrays work
    
    if (typeof dict !== 'object') {
      console.error("makeESFriendly - Unsupported type", typeof (dict));
    }

    if (dict.dataValues != null) {
      // sequelize object detection w/o requiring sequelize dependency
      dict = dict.get();
    }

    const newDict = {};

    for (let key in dict) {
      let skipKey = false;

      if (typeof dict[key] === 'undefined') {
        continue;
      }

      for (const suffix of BANNED_SUFFIX) {
        if (key.endsWith(suffix)) {
          skipKey = true;
        }
      }

      for (const suffix of EPOCH_SUFFIX) {
        if (key.endsWith(suffix)) {
          let tryDate = new Date(dict[key]);
          if (!isNaN(tryDate.getTime())) {
            //is valid date
            dict[key] = tryDate.getTime();
            skipKey = true;
          }
        }
      }

      for (const suffix of DATE_SUFFIX) {
        if (key.endsWith(suffix)) {
          let tryDate = new Date(dict[key]);
          if (!isNaN(tryDate.getTime())) {
            //is valid date
            dict[key] = tryDate.toISOString();
            skipKey = true;
          }
        }
      }

      if (skipKey) {
        newDict[key] = dict[key];
        continue;
      }
      const type = typeof dict[key];

      switch (type) {
        case 'string':
          newDict[`${key}_tx`] = dict[key];
          break;
        case 'number':
          if (isInt(dict[key])) {
            newDict[`${key}_nr`] = dict[key];
            break;
          } else if (isFloat(dict[key])) {
            newDict[`${key}_dec`] = dict[key];
            break;
          } else {
            console.warn('Unknown type? Is number but neither float nor int?', { value: dict[key], "typeof": typeof dict[key] });
          }
        case 'boolean':
          newDict[`${key}_ind`] = dict[key];
          break;
        case 'object':
          if (depth < MAX_DEPTH) {
            if (Array.isArray(dict[key])) {
              newDict[key] = dict[key].map(obj => {
                if (typeof obj === 'object') {
                  return Utils.makeESFriendly(obj, MAX_DEPTH, depth + 1);
                } else {
                  return obj;
                }
              });
            } else if (dict[key]) {
              newDict[key.endsWith('_nest') ? key : `${key}_nest`] = Utils.makeESFriendly(dict[key], MAX_DEPTH, depth + 1);
            }
          } else {
            newDict[key + "_tx"] = JSON.stringify(dict[key], null, 2);
          }
          break;
        case 'undefined':
          break;
        //else fallthrough
        default:
          //not the best default because can cause stuff to fail in watch_min_consumer and that is kinda  hard to find...
          newDict[key] = dict[key];
          break;
      }
    }
    return newDict;
  }
}

module.exports = Utils;
