const Path = require('path');
const Util = require('util');
const Syslogger = require('ain2');
const chalk = require('chalk');
const _console = console;

const Dispatcher = require('../Dispatcher');

const MAX_DEPTH = 2;
/* const MAX_DEPTH = 10; */

const syslog_severity_map = {
  disabled: -1,
  emerg: 0,
  alert: 1,
  crit: 2,
  err: 3,
  error: 3,
  warn: 4,
  notice: 5,
  info: 6,
  debug: 7
};
const SEVERITY_DISABLED = -1;

const color_map = {
  0: chalk.red.bold,
  1: chalk.cyan,
  2: chalk.red.underline,
  3: chalk.red,
  4: chalk.yellow,
  5: chalk.magenta,
  6: chalk.green,
  7: chalk.blue
};

// we need to filter of type string and number otherwise
// util.inspect will convert it to be quoted. i.e: 42 -> '42', hello -> 'hello'
// it's weird
const parseCircular = (obj, depth = MAX_DEPTH) =>
  typeof obj === 'string' || typeof obj === 'number'
    ? obj
    : Util.inspect(obj, { depth: MAX_DEPTH, breakLength: Infinity })
        .split('\n')
        .map(line => line.trim())
        .join(' ');

const LOGGERS = {};

class Logger {

  constructor() {
    this._bind();
    this.name = null;

    this.syslogLevel = SEVERITY_DISABLED;
    this.consoleLevel = SEVERITY_DISABLED;
    this.dispatcherLevel = SEVERITY_DISABLED;

    // sane defaults in case init does not get called
    this.dispatcher = null;
    let consoleDefaults = {
       level: 'info',
       colorize: true,
       timestamp: true
    };
    this._enableConsole(consoleDefaults);

    this.PROJECT_ROOT = process.cwd();
    this.env = "unknown";
    // delay syslog until we have a name
  }

  getLogger(name) {
    if (!name) {
      // root logger is initialized at the bottom
      name = 'root';
    }
    let logger = LOGGERS[name];
    if (logger) {
      return logger;
    }
    logger = new Logger();
    logger.init({ name: name });
    LOGGERS[name] = logger;
    return logger;
  }

  init({ name, env, LOG_SYSLOG, LOG_CONSOLE }, dispatcherOptions) {
    if (name == null) {
      console.error("no name", name, env, LOG_SYSLOG, LOG_CONSOLE);
      throw Error('please initialize logger with the project name');
    }
    this.name = name;
    this.env = env;
    if (LOG_SYSLOG) {
      let syslogOptions = Object.assign({name}, LOG_SYSLOG);
      this._enableSyslog(syslogOptions);
    } 

    if (LOG_CONSOLE) {
      if (typeof LOG_CONSOLE !== 'object') {
        throw Error('options.console must be an object');
      }
      this._enableConsole(LOG_CONSOLE);
    } else {
      this._enableConsole({level: 'disabled'});
    }

    if (dispatcherOptions) {
      this._enableDispatcher(dispatcherOptions);
    }
  }

  _bind() {
    //funcs
    this._send = this._send.bind(this);
    this._getStackInfo = this._getStackInfo.bind(this);
    this._makeLog = this._makeLog.bind(this);
    this._handleErr = this._handleErr.bind(this);
    this.init = this.init.bind(this);
    //base logs
    this.emerg = this.emerg.bind(this);
    this.alert = this.alert.bind(this);
    this.crit = this.crit.bind(this);
    this.error = this.error.bind(this);
    this.warn = this.warn.bind(this);
    this.notice = this.notice.bind(this);
    this.info = this.info.bind(this);
    this.debug = this.debug.bind(this);
    //aliases
    this.emergency = this.emerg;
    this.err = this.error;
    this.warning = this.warn;
    this.log = this.info;
    this.critical = this.crit;
    this.isDebug = this.isDebug.bind(this);
  }

  _handleErr(err) {
    /* process.stdout.write(err); */
    if (err) {
      console.error(`BeboNodeCommons::Logger:_handleErr -- `, err);
    }
  }
  _enableConsole(options) {
    if (typeof options !== 'object') {
      throw Error('options.console must be an object');
    }
    this.consoleSettings = options;
    let lvl = syslog_severity_map[this.consoleSettings.level];
    if (lvl == null) {
      lvl = 7;
    }
    this.consoleLevel = lvl;
    this.lvl = Math.max(this.consoleLevel, this.syslogLevel, this.dispatcherLevel);
  }

  _enableSyslog(LOG_SYSLOG) {
    const tag = LOG_SYSLOG.name;
    if (!tag) {
      throw Error('syslog needs config.syslog.name');
    }
    const facility = LOG_SYSLOG.facility;
    const hostname = LOG_SYSLOG.hostname;
    const address = LOG_SYSLOG.address;
    const level = LOG_SYSLOG.level;

    const options = { address, tag, facility };

    // console.log({ options });

    this.syslogger = new Syslogger();
    this.syslogger.setTransport('UDP');
    this.syslogger.set(options);
    this.hasSyslog = true;
    this.syslogSettings = {
      level,
      hostname
    };
    let lvl = syslog_severity_map[this.syslogSettings.level];
    if (lvl == null) {
      lvl = 7;
    }
    this.syslogLevel = lvl;
    this.lvl = Math.max(this.consoleLevel, this.syslogLevel, this.dispatcherLevel);
  }

  _enableDispatcher(options) {
    if (!options.queueName) {
      throw Error('options.dispatcher requires a queueName property');
    }

    let lvl = syslog_severity_map[options.level];
    if (lvl == null) {
      lvl = 6;
    }
    this.dispatcherLevel = lvl;
    this.lvl = Math.max(this.consoleLevel, this.syslogLevel, this.dispatcherLevel);
    this.dispatcher = Dispatcher.getInstance(options);
  }

  _getStackInfo() {
    const stackIndex =
      this.globalSettings && 'depth' in this.globalSettings ? this.globalSettings.depth : 1;
    // get call stack, and analyze it
    // get all file, method, and line numbers
    const stacklist = new Error().stack.split('\n').slice(3);

    // stack trace format:
    // http://code.google.com/p/v8/wiki/JavaScriptStackTraceApi
    // do not remove the regex expresses to outside of this method (due to a BUG in node.js)
    const stackReg = /at\s+(.*)\s+\((.*):(\d*):(\d*)\)/gi;
    const stackReg2 = /at\s+()(.*):(\d*):(\d*)/gi;

    const s = stacklist[stackIndex] || stacklist[0];
    const sp = stackReg.exec(s) || stackReg2.exec(s);

    if (sp && sp.length === 5) {
      return {
        method: sp[1],
        relativePath: Path.relative(this.PROJECT_ROOT, sp[2]),
        line: sp[3],
        pos: sp[4],
        file: Path.basename(sp[2]),
        stack: stacklist.join('\n')
      };
    }
    console.error('Logger can not parse stack list', stacklist);
    return {
      method: '',
      relativePath: '',
      line: '',
      pos: '',
      file: '',
      stack: ''
    };
  }

  _makeLog(level, str, stackInfo, now, syslog = false) {
    const severity = syslog_severity_map[level];
    const chalkFn = color_map[severity];

    let msg = '';

    if (this.consoleSettings && this.consoleSettings.timestamp === true && !syslog) {
      //auto-added, at least on my local when I test
      msg += now + ' -';
    }

    msg += ' ';

    if (this.consoleSettings && this.consoleSettings.colorize === true && !syslog) {
      msg += chalkFn(level);
    } else {
      msg += level;
    }

    msg += ' ';
    msg += `[${stackInfo.relativePath}:${stackInfo.line}:${stackInfo.pos}] `;
    msg += str;
    msg += '\n';

    return msg;
  }

  _send(level, args) {
    const arr = Object.keys(args).map(i => args[i]);
    const str = arr.map(arg => parseCircular(arg)).join(' ');

    const severity = syslog_severity_map[level];
    const stackInfo = this._getStackInfo();
    const now = new Date().toISOString();

    if (severity <= this.syslogLevel) {
      this.syslogger.send(
        this._makeLog(level, str, stackInfo, now, true),
        severity,
        this._handleErr
      );
    }

    if (severity <= this.dispatcherLevel) {
      const event = {
        level_tx: level,
        message_tx: str,
        message_sa: str,
        event_dttm: now,
        relative_path_tx: stackInfo.relativePath,
        line_nr: stackInfo.line,
        pos_nr: stackInfo.pos,
        method_tx: stackInfo.method,
        file_tx: stackInfo.file
      };
      this.dispatcher.writeEvent(event);
    }

    if (severity <= this.consoleLevel) {
      process.stdout.write(this._makeLog(level, str, stackInfo, now, false), this._handleErr);
    }
  }

  emerg() {
    this._send('emerg', arguments);
  }
  alert() {
    this._send('alert', arguments);
  }
  crit() {
    this._send('crit', arguments);
  }
  error() {
    this._send('error', arguments);
  }
  warn() {
    this._send('warn', arguments);
  }
  notice() {
    this._send('notice', arguments);
  }
  info() {
    this._send('info', arguments);
  }
  debug() {
    this._send('debug', arguments);
  }
  isDebug() {
    return this.lvl  >= syslog_severity_map.debug;
  }
}

const RootLogger = new Logger();
// root logger has no name until the application gives it one
LOGGERS['root'] = RootLogger;
module.exports = RootLogger;
