const Promise = require('bluebird');
const AWS = require('aws-sdk');

AWS.config.setPromisesDependency(Promise);

const config = require('../Config');
const default_log = require('../Logger');

const SQSWorker = require('./sqs_worker');

const MAX_RETRIES = 10;
const RETRY_DELAY_IN_SECONDS = 10;

const getQueueArnFromQueueName = (aws_region, queueName) =>
  `arn:aws:sqs:${aws_region}:302024746872:${queueName}`;

const getQueueUrlFromQueueName = (aws_region, queueName) =>
  `https://sqs.${aws_region}.amazonaws.com/302024746872/${queueName}`;

const getQueueNameFromQueueUrl = queueUrl => {
  const splitQueueUrl = queueUrl.split('/');
  return splitQueueUrl[splitQueueUrl.length -1];
};


class RetryError extends Error {

  constructor(message) {
    if (message == null) {
      message = "RetryError";
    }
    super(message);
    this.name = "RetryError";
  }
}

class SQS {

  constructor(options={}) {
    if (options.log) {
      this.log = options.log;
    } else {
      this.log = default_log;
    }

    if (options.queueName) {
      this.queueName = options.queueName;
    } else {
      this.queueName = `${config.REGION}_${config.SERVICE}_${config.HOSTNAME}`;
    }

    this.sqs = null;
    this.sns = new AWS.SNS({ apiVersion: '2010-03-31', region: config.SNS_DESTINATION_REGION });
    this.handlers = [];
    this.sqsWorker = null;

    this.init_called = false;
    this.init_done = false;

    this.init = this.init.bind(this);
    this.onMessage = this.onMessage.bind(this);
    this.sqsWorkerOnError = this.sqsWorkerOnError.bind(this);
    this.registerHandler = this.registerHandler.bind(this);
    this.start = this.start.bind(this);
    this.createQueueIfNotExists = this.createQueueIfNotExists.bind(this);
    this.doesQueueExists = this.doesQueueExists.bind(this);
    this.createQueue = this.createQueue.bind(this);
    this.subscribeSQSToSNS = this.subscribeSQSToSNS.bind(this);

    this.init();
  }

  init() {
    if (this.init_called) {
      if (this.init_done) {
        return Promise.resolve();
      }
      return Promise.delay(100).then(() => this.init());
    }

    this.init_called = true;

    return config.getAwsRegion()
      .then(aws_region => {
        this.AWS_REGION = aws_region;
        this.sqs = this.sqs || new AWS.SQS({ apiVersion: '2012-11-15', region: this.AWS_REGION, httpOptions: { timeout: 30 * 1000, connectTimeout: 10 * 1000 }});
        this.queueUrl = getQueueUrlFromQueueName(this.AWS_REGION, this.queueName);
        this.queueArn = getQueueArnFromQueueName(this.AWS_REGION, this.queueName);
      })
      .then(() => this.createQueueIfNotExists())
      .then(() => {
        this.sqsWorker = new SQSWorker(this.queueUrl, this.sqs, this.onMessage, this.log);
        this.sqsWorker.onError = this.sqsWorkerOnError;
        this.init_done = true;
      });
  }

  async onMessage(message) {
    for (let handler of this.handlers) {
      await handler(message);
    }
  }

  sqsWorkerOnError(err) {
    this.log.error('SQSWorker', err);
    if (err && err.code && err.code === 'AWS.SimpleQueueService.NonExistentQueue') {
      return this.createQueueIfNotExists();
    }
    return Promise.delay(100);
  }

  registerHandler(handler) {
    this.handlers.push(handler);
  }

  start() {
    this.init()
      .then(() => this.sqsWorker.start());
  }

  createQueueIfNotExists() {
    return this.doesQueueExists()
      .then(queueExists => {
        this.log.debug('createQueueIfNotExists -- queueExists?', queueExists);
        if (queueExists) {
          return Promise.resolve();
        }
        return this.createQueue();
      })
      .then(() => this.queueUrl);
  }

  doesQueueExists() {
    this.log.info(`Checking if SQS queue ${this.queueName} exists`);
    return this.sqs.listQueues({
      QueueNamePrefix: this.queueName.substring(this.queueName.length -1, 0) // AWS SQS is bullshit -- if QueueNamePrefix is the queueName, it doesn't match. :rolling_eyes:
    })
    .promise()
    .then(listQueuesResponse => {
      if (listQueuesResponse.QueueUrls) {
        const queueNames = listQueuesResponse.QueueUrls.map(getQueueNameFromQueueUrl);
        const queueIndex = queueNames.indexOf(this.queueName);
        if (queueIndex > -1) {
          return true;
        }
      }
      return false;
    })
    .catch(err => {
      this.log.error(`Failed to check if SQS queue ${this.queueName} exists`, err);
      return false;
    });
  }

  createQueue(tries=0) {
    if (tries == MAX_RETRIES) {
      return Promise.reject('max_retries_exceeded');
    }
    return this.sqs.createQueue({
      QueueName: this.queueName,
      Attributes: {
        Policy: JSON.stringify({
          "Version": "2012-10-17",
          "Id": `${this.queueArn}/SQSDefaultPolicy`,
          "Statement": [
            {
              "Sid": `Sid${Date.now()}`,
              "Effect": "Allow",
              "Principal": {
                "AWS": "*"
              },
              "Action": "SQS:SendMessage",
              "Resource": this.queueArn,
              "Condition": {
                "ArnEquals": {
                  "aws:SourceArn": config.SNS_TOPIC_ARN
                }
              }
            }
          ]
        })
      }
    })
    .promise()
    .then(createQueueResponse => {
      if (createQueueResponse && createQueueResponse.QueueUrl) {
        return this.subscribeSQSToSNS();
      } else {
        return Promise.reject('invalid_response');
      }
    })
    .catch(err => {
      const delay = (err.retryDelay || RETRY_DELAY_IN_SECONDS) * 1000;
      this.log.error('Failed to createQueue -- retrying in', delay, err);
      return Promise.delay(delay).then(() => this.createQueue(tries + 1));
    });
  }

  subscribeSQSToSNS() {
    this.log.info(`Subscribing ${this.queueArn} to ${config.SNS_TOPIC_ARN}`);
    const subscribeParams = {
      Protocol: 'sqs',
      TopicArn: config.SNS_TOPIC_ARN,
      Endpoint: this.queueArn
    };
    return this.sns.subscribe(subscribeParams)
      .promise()
      .then(subscriptionResponse => {
        const subscriptionArn = subscriptionResponse.SubscriptionArn;
        this.log.info(`created subscription from ${config.SNS_TOPIC_ARN} to ${this.queueArn} -- subscriptionArn: ${subscriptionArn}`);
      });
  }

}

SQS.RetryError = RetryError;
module.exports = SQS;
