const Promise = require('bluebird');
const uuid = require('uuid');

class SQSWorker {
  constructor(queueUrl, sqs, onMessage, log) {

    this.name = '[SQSWorker]';
    this.queueUrl = queueUrl;
    this.onMessage = onMessage;
    this.sqs = sqs;
    this.log = log;

    this.receiveMessagesFromSQS = this.receiveMessagesFromSQS.bind(this);
    this.deleteMessagesFromSQS = this.deleteMessagesFromSQS.bind(this);

  }

  start() {
    return this.receiveMessagesFromSQS(this.queueUrl)
      .then(() => this.start());
  }

  receiveMessagesFromSQS(queueUrl) {
    const receiveParams = {
      QueueUrl: queueUrl,
      MaxNumberOfMessages: 10,
      VisibilityTimeout: 3, // FIXME - expose on api
      WaitTimeSeconds: 20
    };
    return this.sqs.receiveMessage(receiveParams)
      .promise()
      .then(async (receiveMessageResponse) => {

        const receivedMessages = receiveMessageResponse.Messages;
        const messageReceipts = [];
        if (!receivedMessages) {
          return [];
        }

        const tuples = [];

        for (const message of receivedMessages) {
          try {
            const messageBodyAsJSON = JSON.parse(message.Body);
            const SNSMessageAsJSON = JSON.parse(messageBodyAsJSON.Message);
            let p = this.onMessage(SNSMessageAsJSON);
            tuples.push([p, message.ReceiptHandle, SNSMessageAsJSON]);
          } catch(err) {
            this.log.error(this.name, 'Failed to parse message body to JSON', err, message);
          }
        }
        for (let t of tuples) {
          let p = t[0];
          let handle = t[1];
          if (p == null) {
            messageReceipts.push(handle);
          } else {
            try {
              await p;
              messageReceipts.push(handle);
            } catch (error) {
              if (error && error.name === "RetryError") {
                // noop - we retry the message later
              } else {
                this.log.error(error, error && error.stack, t[2]);
              }
            }
          }
        }
        return messageReceipts.map(receiptHandle => ({ ReceiptHandle: receiptHandle, Id: uuid.v4() }));
      })
      .then(entries => {
        this.deleteMessagesFromSQS(queueUrl, entries);
        return Promise.resolve();
      })
      .catch(this.onError);
  }

  deleteMessagesFromSQS(queueUrl, entries) {
    if (!entries || !entries.length) {
      return;
    }
    const deleteMessageBatchParams = {
      QueueUrl: queueUrl,
      Entries: entries
    };
    return this.sqs.deleteMessageBatch(deleteMessageBatchParams)
      .promise();
  }
  onError(error) {
    this.log.warn(this.name, 'onError SHOULD BE OVERWRITTEN', error);
  }
}

module.exports = SQSWorker;
