const request = require('request-promise');
const Promise = require('bluebird');

const logger = require('../Logger');
const config = require('../Config');
const error = require('../Error');

const defaultOptions = {
  timeout: 20000,
  json: true
};

function checkResponse(res) {
  if (res.code === 200 || res.code === 201) return Promise.resolve(res);
  else if (res.code === 400) return Promise.reject(new error.BadRequest(res));
  else if (res.code === 401) return Promise.reject(new error.Unauthorized(res));
  else if (res.code === 404) return Promise.resolve(null);
  else if (res.code === 500) return Promise.reject(new error.InternalServerError(res));
  else return Promise.reject(new error.DefaultError(res));
}

function addCacheKey(user) {
  if (!user) return user;
  user.from_cache = false;
  return user;
}

module.exports = {
  getUser: (user_id = null, access_token = null) => {
    let options = Object.assign({}, defaultOptions, { url: config.aws_api_url + '/user/me' });

    options['headers'] = access_token
      ? { 'X-Access-Token': access_token }
      : { 'X-Api-Key': config.SERVICE_API_KEY, 'x-user-id': user_id };

    return request(options)
      .then(addCacheKey)
      .then(checkResponse)
      .then(user => user.result[0]);
  },
  updateUser: (user_id, update) => {
    if (!update.user_id) update['user_id'] = user_id;

    let options = Object.assign({}, defaultOptions, {
      method: 'PUT',
      url: config.aws_api_url + '/user/' + user_id,
      body: update
    });

    options['headers'] = {
      'X-Api-Key': config.SERVICE_API_KEY
    };

    return request(options)
      .then(checkResponse)
      .then(user => user.result[0]);
  }
};
