const Promise = require('bluebird');

const api = require('./api.js');
const cache = require('../RedisCache');
const AUTH_REDIS_DB = require('../Config').AUTH_REDIS_DB;
const error = require('../Error');

const EXPIRE_TIME = 60 * 60 * 3;

module.exports = {
  // gets user from user_id
  get: user_id => {
    return cache
      .getReadClient(AUTH_REDIS_DB)
      .getAsync(user_id)
      .then(user => {
        return user ? JSON.parse(user) : api.getUser(user_id);
      })
      .then(user => {
        if (!user) return null;
        if (user.from_cache) return user;

        user.from_cache = true;
        return cache
          .getWriteClient(AUTH_REDIS_DB)
          .setAsync(user_id, JSON.stringify(user), 'EX', EXPIRE_TIME)
          .then(() => {
            user.from_cache = false;
            return user;
          });
      })
      .catch(err => {
        return Promise.reject(err);
      });
  },
  // gets access_token from cache, otherwise get it from JAVA api and set it in cache for one hour
  getByAccessToken: access_token => {
    return cache
      .getReadClient(AUTH_REDIS_DB)
      .getAsync(access_token)
      .then(user_id => {
        return user_id ? cache.getReadClient(AUTH_REDIS_DB).getAsync(user_id) : null;
      })
      .then(user => {
        return user ? JSON.parse(user) : api.getUser(null, access_token);
      })
      .then(user => {
        if (!user) return null;
        if (user.from_cache) return user;

        user.from_cache = true;
        return Promise.all([
          cache
            .getWriteClient(AUTH_REDIS_DB)
            .setAsync(access_token, user.user_id, 'EX', EXPIRE_TIME),
          cache
            .getWriteClient(AUTH_REDIS_DB)
            .setAsync(user.user_id, JSON.stringify(user), 'EX', EXPIRE_TIME)
        ]).then(() => {
          user.from_cache = false;
          return user;
        });
      })
      .catch(err => {
        return Promise.reject(err);
      });
  },
  update: (user_id, update) => {
    return api
      .updateUser(user_id, update)
      .then(res => {
        return cache
          .getWriteClient(AUTH_REDIS_DB)
          .DELAsync(user_id)
          .then(() => res);
      })
      .catch(err => {
        return Promise.reject(err);
      });
  }
};
