import os
import sys
import syslog
import time
import traceback
import socket

from math import inf

import logging.handlers
from colorlog import ColoredFormatter

#  from bebo.config import options
from bebo.dispatcher import write as dispatcher_write

# For pretty log messages, if available

try:
    import curses
except ImportError:
    curses = None

logger_map = {}

#  time hostname proc_name[pid]: level [(thread\:)?file:line:column] msg

SYSLOG_LOG_FMT = '%(levelname)s [%(threadName)s:%(module)s.py:%(lineno)d] %(message)s'
COLOR_LOG_FMT = ('%(log_color)s[%(levelname)1.1s %(asctime)s %(threadName)s:%(module)s.py:%(lineno)d'
                 '] %(reset)s%(message)s')

# Use different local facilities by application, default to local1 as a catchall
FACILITIES = {'local0': logging.handlers.SysLogHandler.LOG_LOCAL0, 'local3': logging.handlers.SysLogHandler.LOG_LOCAL3,
              'local1': logging.handlers.SysLogHandler.LOG_LOCAL1, 'local4': logging.handlers.SysLogHandler.LOG_LOCAL4,
              'local2': logging.handlers.SysLogHandler.LOG_LOCAL2, 'local5': logging.handlers.SysLogHandler.LOG_LOCAL5,
              'local6': logging.handlers.SysLogHandler.LOG_LOCAL6, 'local7': logging.handlers.SysLogHandler.LOG_LOCAL7}


class WildlingFormatter(logging.Formatter):
    def format(self, record):
        record_dict = record.__dict__
        #  del record_dict.created
        #  del record_dict.args
        #  del record_dict.relativeCreated
        #  del record_dict.process
        #  del record_dict.msecs
        #  del record_dict.thread
        #  del record_dict.levelno
        if record_dict.get('exc_info', None):
            record_dict['exc_info'] = list(map(str, record_dict['exc_info']))
        record_dict['levelname'] = record_dict['levelname'].lower()
        return record_dict

class WildlingHandler(logging.Handler):
    def __init__(self, dispatcher_opts):
        logging.Handler.__init__(self)
        self.dispatcher_opts = dispatcher_opts

    def emit(self, record):
        log_entry = self.format(record)
        if self.dispatcher_opts['log_level'] <= record.levelno:
            dispatcher_write('{}-logs_es'.format(record.name), log_entry, self.dispatcher_opts['ttl'])

# Monkey patch the logging formatter to print the full stack trace
# http://www.velocityreviews.com/forums/t717977-castrated-traceback-in-sys-exc_info.html
# - PDW
def formatException(_, ei):
    """
    Format and return the specified exception information as a string.
    This implementation builds the complete stack trace, combining
    traceback.format_exception and traceback.format_stack.
    """
    lines = traceback.format_exception(*ei)
    if ei[2]:
        lines[1:1] = traceback.format_stack(ei[2].tb_frame.f_back)
    return ''.join(lines)


# monkey patch the logging module
logging.Formatter.formatException = formatException


def add_console(logger, level, fmt=COLOR_LOG_FMT):
    channel = logging.StreamHandler()
    channel.setLevel(level)
    log_colors = {'CRITICAL': 'bold_red',
                  'DEBUG': 'blue',
                  'ERROR': 'red',
                  'INFO': 'green',
                  'WARNING': 'yellow'}

    channel.setFormatter(ColoredFormatter(fmt, datefmt="%y%m%d %H:%M:%S.", reset=True, log_colors=log_colors))
    logger.addHandler(channel)


def add_syslog(logger, address, level, ident=os.path.basename(sys.argv[0]).replace('.py', ''), facility='local1', fmt=SYSLOG_LOG_FMT):
    # Root logger logs to syslog
    if facility in FACILITIES:
        sys_facility = FACILITIES[facility]
    else:
        sys_facility = FACILITIES['local1']

    try:
        socket.gethostbyaddr(address[0])
    except socket.gaierror:
        print('Failed to create syslog at specified address due to resolve error -- not adding syslog')
        return

    channel = logging.handlers.SysLogHandler(address, sys_facility, socket.SOCK_DGRAM)
    channel.ident = socket.getfqdn().split('.')[0] + ' ' + ident + '[' + str(os.getpid()) + ']:'
    channel.setLevel(level)
    syslog_formatter = logging.Formatter(fmt)
    channel.setFormatter(syslog_formatter)
    logger.addHandler(channel)

def add_wildling(logger, dispatcher_opts):
    dispatcher_opts['log_level'] = log_level(dispatcher_opts.get('log_level', 'info'))
    dispatcher_opts['ttl'] = dispatcher_opts.get('ttl', 60)
    print('adding wildling handler', dispatcher_opts)
    wildling_handler = WildlingHandler(dispatcher_opts)
    wildling_handler.setFormatter(WildlingFormatter())
    logger.addHandler(wildling_handler)

def initialize(
        name,
        console_log,
        syslog_log,
        syslog_level,
        console_level,
        dispatcher_opts
    ):

    logger = logging.getLogger(name)

    # TODO check this hack out:
    os.environ['TZ'] = "Etc/UTC"
    time.tzset()

    if syslog_log is not True and console_log is not True:
        print('BeboPythonCommons::initialize failed due to invalid config -- need either syslog_log or console_log to be True. syslog_log: {}, console_log: {}'.format(syslog_log, console_log))
        return

    if syslog_log:
        syslog_addr = ('localhost', 514)
        print('creating syslogger', syslog_addr)
        add_syslog(logger, syslog_addr, log_level(syslog_level), ident=name, facility=FACILITIES['local4'], fmt=SYSLOG_LOG_FMT)

    if console_log:
        add_console(logger, log_level(console_level), COLOR_LOG_FMT)

    if dispatcher_opts:
        add_wildling(logger, dispatcher_opts)

    # Set logger to min of console/syslog to log debug if requested
    logger.setLevel(min(log_level(syslog_level), log_level(console_level)))
    return logger

def log_level(level):
    return {'debug': logging.DEBUG, 'info': logging.INFO, 'warn': logging.WARN, 'warning': logging.WARNING,
            'error': logging.ERROR, 'critical': logging.CRITICAL, 'fatal': logging.FATAL}[level]

def get_logger(
        name,
        CONSOLE_LOG=True,
        SYSLOG_LOG=False,
        SYSLOG_LEVEL='info',
        CONSOLE_LEVEL='info',
        dispatcher_opts=None
    ):

    if name in logger_map:
        return logger_map[name]

    logger = initialize(
        name=name,
        console_log=CONSOLE_LOG,
        syslog_log=SYSLOG_LOG,
        syslog_level=SYSLOG_LEVEL,
        console_level=CONSOLE_LEVEL,
        dispatcher_opts=dispatcher_opts
    )
    logger_map[name] = logger
    return logger

CRITICAL = logging.CRITICAL
DEBUG = logging.DEBUG
ERROR = logging.ERROR
INFO = logging.INFO
NOTSET = logging.NOTSET
WARNING = logging.WARNING

if __name__ == "__main__":
    test_logger = get_logger('bebopythoncommmons_logger', SYSLOG_LOG=True, dispatcher_opts={'log_level': 'info'})
    test_logger.debug('debug')
    test_logger.info('info')
    test_logger.warn('warn')
    test_logger.error('error')
