#!/usr/bin/env python3
# vim set sw=4 ts=4 et :

import argparse
import sys
import subprocess
import logging
import os
import requests
import time

from terraform.config import cfg

logging.basicConfig(level=logging.INFO,
        format='%(asctime)s %(levelname)s %(module)s - %(message)s', datefmt="%Y-%m-%d %H:%M:%S")

JENKINS_TOKEN = '9Zv3rfIncGFC'
JENKINS_HOST = 'jenkins-stretch.bebo-dev.com'
JENKINS_PROJECT = 'bebo-terraform'

class Deploy:
    def __init__(self, action, env, tag, passed_region, vpc_id):
        if action == 'create':
            self.jenkins_project = 'bebo-terraform'
        else:
            self.jenkins_project = 'bebo-terraform-destroy'

        self.env = env
        self.tag = tag

        self.session = requests.Session()
        self.isCsrf = None

        repo = 'git@github.com:bebo/%s' % JENKINS_PROJECT
        subprocess.call(['git', 'pull'])

        if self.tag == 'none':
            self.tag = self.__create_git_tag()

        if action == 'create':
            if passed_region != 'none':
                vpc_id = get_vpc_id(passed_region) if not vpc_id else vpc_id
                self._build_project(vpc_id, passed_region, self.env)
            else:
                for aws_region in cfg.aws_regions:
                    vpc_id = get_vpc_id(aws_region)
                    self._build_project(vpc_id, aws_region, self.env)
            return

        if vpc_id == 'none':
            logging.error('vpc-id is required if calling destroy')
            return

        try:
            vpc_ids = vpc_id.split(',')
            for _vpc_id in vpc_ids:
                if _vpc_id:
                    self._build_project(_vpc_id)
                else:
                    logging.warn('Not destroying "%s" because it isnt truthy...' % _vpc_id)

        except Exception as e:
            logging.exception('Failed to destroy vpc %s' % e)


    def __updateCsrf(self):
        try:
            url = "https://{}/crumbIssuer/api/json".format(JENKINS_HOST)

            res = self.session.get(url)

            if res.status_code == 404:
                logging.info("The CSRF protection was disabled by the server.")
                self.isCsrf = False
                return
            else:
                res.raise_for_status()

            crumb = res.json()
            self.session.headers.update({crumb["crumbRequestField"]: crumb["crumb"]})
            self.isCsrf = True

        except Exception as e:
            logging.error("Fail to update CSRF!: {}".format(e))
            raise

    def __create_git_tag(self):
        process = subprocess.Popen(['git', 'rev-parse', '--abbrev-ref', 'HEAD'], stdout=subprocess.PIPE)
        out, err = process.communicate()
        branch = out.decode('utf-8')[:-1:]
        tag_name = '{}-{}'.format(branch, int(time.time()))
        subprocess.call(['git', 'tag', tag_name])
        logging.info('Tag not specified. Creating new tag: {}'.format(tag_name))
        subprocess.call(['git', 'push'])
        logging.info('Created and pushed new tag: ' + tag_name)
        return tag_name

    def _build_project(self, vpc_id, passed_region=None, env=None):
        url = 'https://{}/job/{}/buildWithParameters?token={}&TAG={}&VPC_ID={}'.format(JENKINS_HOST, self.jenkins_project, JENKINS_TOKEN, self.tag, vpc_id.strip())
        if env:
            url += '&BEBO_ENV={}'.format(self.env)
        if passed_region:
            url += '&AWS_REGION={}'.format(passed_region)

        print(url)

        if self.isCsrf is None or self.isCsrf:
            self.__updateCsrf()

        request = self.session.post(url)
        print(request.text)


def get_vpc_id(passed_region):
    command = "python terraform/terraform.py --aws-region %s gen_id" % passed_region

    commands = command.split(" ")

    print(commands)

    proc = subprocess.Popen(commands, stdout=subprocess.PIPE, cwd=os.getcwd())
    out, err = proc.communicate()

    if err is not None:
        print("ERR: %s" % err)
        exit(1)

    split = out.split(b"\n")
    l = split[2].rstrip().decode('utf-8')
    id = l.split(":")[1]

    return id

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('action', choices=['create', 'destroy'],
                        help='Action to take. One of [create, destroy]')
    parser.add_argument('-e', '--env',
                        help='env to deploy: dev, ptr, prod')
    parser.add_argument('-t', '--tag', default='none',
                        help='The tag to deploy.  If not provided, a new tag will be created')
    parser.add_argument('-r', '--region', default='none',
            help='AWS Region to deploy to. NOTE: Overrides default regions for env')
    parser.add_argument('--vpc-id', default='none',
            help='VPC ID To create. If not provided, a new id will be created')

    args = parser.parse_args()

    if args.action == 'create' and not args.env:
        logging.error('error: the following arguments are required for non-destroy runs: -e/--env')
        sys.exit(1)

    print(args)

    Deploy(action=args.action, env=args.env, tag=args.tag, passed_region=args.region, vpc_id=args.vpc_id)

if __name__ == '__main__':
    main()
