import pystache
import os
import sys

from config import cfg

from ec2_controller import EC2Controller
from terraform_controller import TerraformController
from bucket_policy_manager import BucketPolicyManager
from route53_controller import Route53Controller
from monitoring_controller import MonitoringController

from config import cfg

tfvars_template = pystache.parse(open("tfvars_template.mustache").read())
renderer = pystache.Renderer()

class Controller:
    def __init__(self, aws_region, vpc_id, do_prompt, force, bebo_env, dry_run):
        self.aws_region = aws_region
        self.vpc_id = vpc_id
        self.do_prompt = do_prompt
        self.force = force
        self.bebo_env = bebo_env
        self.dry_run = dry_run

        self.files = cfg.BASE_TF_FILES

        self.ec2_controller = EC2Controller(self.aws_region)
        if self.bebo_env is None:
            self.bebo_env = self.ec2_controller.get_env_from_vpc(self.vpc_id)

        self.bucket_policy_manager = BucketPolicyManager(self.aws_region, self.ec2_controller)
        self.route53_controller = Route53Controller(self.aws_region, self.ec2_controller)
        self.terraform_controller = TerraformController(self.aws_region, self.bebo_env, self.vpc_id, self.ec2_controller, self.files)
        self.monitoring_controller = MonitoringController(self.vpc_id, self.bebo_env)

    def act(self, command):
        print("Controller acting on \"%s\"" % command)
        print("BEBO_ENV: %s" % self.bebo_env)
        print("VPC_ID: %s" % self.vpc_id)
        if command == "create":
            self.create()
        elif command == "destroy":
            self.destroy()
        elif command == "gen_id":
            pass
        else:
            print("Unknown command %s" % command)

    def create(self):
        vpc_exists = self.ec2_controller.check_vpc_exists(self.vpc_id)
        if vpc_exists is True and self.force is False:
            print("not a new vpc, exiting")
            sys.exit(1)

        tfvars_data = self.terraform_controller.create_tfvars_data()
        print("TFVars data:",  tfvars_data)
        if self.do_prompt is True:
            should_continue = input("Continue? \"yes\" or other only: \n") == "yes"
        else:
            should_continue = True

        if should_continue is False:
            print("aborted")
            sys.exit(0)

        _dir = self.write_new_vpc_to_fs(tfvars_data)
        good = self.terraform_controller.create(self.do_prompt, self.dry_run)
        print("TerraformController.create() done... is good? %s" % good)

        if good is not True:
            print("TerraformController is not good -- exiting")
            sys.exit(1)

        if self.dry_run == True:
            print("exiting because dry_run")
            sys.exit(0)

        if self.ec2_controller.check_vpc_exists(self.vpc_id) is not True:
            print("VPC Doen't exist after TerraformController.create -- exiting")
            sys.exit(1)

        vpc = self.ec2_controller.get_aws_vpc_by_vpc_id(self.vpc_id)[0]
        #  self.bucket_policy_manager.add_vpc_to_policy(vpc)
        self.route53_controller.add_vpc_to_internal_routing(vpc, self.vpc_id, cfg.INTERNAL_DNS_ZONE_ID)
        self.monitoring_controller.add_monitoring()

    def destroy(self):
        vpcs = self.ec2_controller.get_aws_vpc_by_vpc_id(self.vpc_id)
        if len(vpcs) > 0:
            vpc = vpcs[0]
        elif self.force is False:
            print("vpc doesn't exist, exiting")
            return
        else:
            vpc = None

        self.monitoring_controller.remove_monitoring()
        print("monitoring removed")

        good = self.terraform_controller.destroy(self.do_prompt)
        print("TerraformController.destroy() done... is good? %s" % good)
        if good is not True:
            print("TerraformController is not good -- exiting")
            sys.exit(1)

        if vpc is not None:
            aws_vpc_id = vpc['VpcId']
            self.bucket_policy_manager.remove_vpc_from_policy(aws_vpc_id)
            self.route53_controller.remove_vpc_from_internal_routing(self.vpc_id, aws_vpc_id, cfg.INTERNAL_DNS_ZONE_ID)


    def write_new_vpc_to_fs(self, tfvars_data):
        vpc_dir = "./vpcs/%s" % self.vpc_id

        try:
            os.mkdir(vpc_dir)
        except FileExistsError:
            pass

        with open("%s/terraform.tfvars" % vpc_dir, "w") as fw:
            tfvars_file = renderer.render(tfvars_template, tfvars_data)
            fw.write(tfvars_file)
            fw.close()

        return vpc_dir

