from datetime import datetime
import boto3

class EC2Controller:
    def __init__(self, aws_region):
        self.aws_region = aws_region
        self.ec2 = boto3.resource('ec2', region_name=aws_region)
        self.ec2_client = self.ec2.meta.client

    def get_vpc_endpoint(self, vpc_id):
        vpces = self.ec2_client.describe_vpc_endpoints(
                Filters=[
                    {
                        'Name': 'vpc-id',
                        'Values': [
                            vpc_id
                            ]
                        }
                    ]
                )['VpcEndpoints']

        if len(vpces) == 0:
            return None
        else:
            return vpces[0]

    def get_aws_vpc_by_vpc_id(self, vpc_id):
        vpcs = self.ec2_client.describe_vpcs(
                Filters=[
                    {
                        'Name': 'tag:Name',
                        'Values': [
                            vpc_id
                            ]
                        },
                    ]
                ).get('Vpcs', list())
        return vpcs

    def check_vpc_exists(self, vpc_id):
        vpcs = self.get_aws_vpc_by_vpc_id(vpc_id)
        if len(vpcs) == 1:
            return True
        elif len(vpcs) > 1:
            print("WARNING: >1 VPC w/ vpc_id: %s?" % vpc_id)
            return False
        else:
            return False

    def get_aws_zones(self, aws_region):

        zones_req = self.ec2_client.describe_availability_zones(
            Filters=[
                {
                    'Name': 'region-name',
                    'Values': [
                        aws_region
                        ]
                    }
                ]
            )

        def convert_to_mustache_format(zone):
            return {
                'zone': zone,
                'comma': True
                }
        zones = [zone['ZoneName'] for zone in zones_req["AvailabilityZones"]]
        print(zones)
        if aws_region == 'ap-northeast-1' and 'ap-northeast-1b' in zones:
            zones.remove('ap-northeast-1b')
        elif aws_region == 'sa-east-1' and 'sa-east-1b' in zones:
            zones.remove('sa-east-1b')

        zones = [convert_to_mustache_format(zone) for zone in zones]

        zones[-1]['comma'] = False
        return zones

    def get_core_ami(self):
#TODO add active filtering to packer then add below
        images = self.ec2_client.describe_images(
                Filters=[
                    {
                        'Name': 'tag:build_name',
                        'Values': ['core']
                        }
                    ]
                )["Images"]
        winning_ami = images[0]
        for ami in images:
            if datetime.strptime(ami["CreationDate"], '%Y-%m-%dT%H:%M:%S.%fZ') > datetime.strptime(winning_ami["CreationDate"], '%Y-%m-%dT%H:%M:%S.%fZ'):
                winning_ami = ami
        return winning_ami["ImageId"]

    def get_composer_ami(self):
#TODO add active filtering to packer then add below
        images = self.ec2_client.describe_images(
                Filters=[
                    {
                        'Name': 'tag:build_name',
                        'Values': ['composer']
                        }
                    ]
                )["Images"]
        winning_ami = images[0]
        for ami in images:
            if datetime.strptime(ami["CreationDate"], '%Y-%m-%dT%H:%M:%S.%fZ') > datetime.strptime(winning_ami["CreationDate"], '%Y-%m-%dT%H:%M:%S.%fZ'):
                winning_ami = ami
        return winning_ami["ImageId"]


    def terminate_extra_instances(self, vpc_id):
        vpcs = self.get_aws_vpc_by_vpc_id(vpc_id)

        if len(vpcs) == 0:
            print("ERROR: VPC doesn't exist?")
            return

        aws_vpc_id = vpcs[0]['VpcId']

        print("aws_vpc_id to terminate instances from: %s" % aws_vpc_id)

        reservations = self.ec2_client.describe_instances(
                Filters=[
                    {
                        'Name': 'vpc-id',
                        'Values': [
                            aws_vpc_id
                            ]
                        },
                    {
                        'Name': 'instance-state-name',
                        'Values': [
                            'running', 'pending', 'stopped'
                            ]
                        }
                    ]
                ).get('Reservations', list())

        ids = list()

        for res in reservations:
            for instance in res["Instances"]:
                id = instance["InstanceId"]
                hostname = None
                for tag in instance.get("Tags", list()):
                    if tag["Key"] == "Name":
                        hostname = tag["Value"]

                if hostname is not None and "-admin-" in hostname:
                    continue

                print("Terminating instance: id: %s, hostname: %s" % (id, hostname))
                ids.append(id)

        if len(ids) == 0:
            print("No extra instances to terminate in %s? Was this VPC successful?" % vpc_id)
        else:
            self.ec2_client.terminate_instances(
                    InstanceIds=ids
                )

    def get_env_from_vpc(self, vpc_id):
        vpcs = self.get_aws_vpc_by_vpc_id(vpc_id)
        if not vpcs:
            return None

        tags = {}
        for vpc in vpcs:
            if len(tags):
                break
            local_tags = {}
            for tag in vpc['Tags']:
                local_tags[tag['Key']] = tag['Value']

            if 'Name' in local_tags and local_tags['Name'] == vpc_id:
                tags = local_tags

        return tags.get('BEBO_ENV', 'dev')

