import boto3
import json
import sys
import argparse
from botocore.exceptions import ClientError

PARSER = argparse.ArgumentParser(description="Add self to regional S3 Bucket")
PARSER.add_argument("--aws-region", "-r", dest="aws_region", help="the AWS Region to add to", required=True)
PARSER.add_argument("--vpc-id", dest="vpc_id", help="AWS recognised VPC ID")

def convert_aws_region_to_short(aws_region):
    split = aws_region.split("-")
    prefix = split[0]
    number = split[-1]
    modifier = ""
    if "north" in split[1]:
        modifier += "n"
    if "south" in split[1]:
        modifier += "s"
    if "east" in split[1]:
        modifier += "e"
    if "west" in split[1]:
        modifier += "w"
    if "central" in split[1]:
        modifier += "c"

    end = prefix + modifier + number

    #  end = split[0] + split[1][0] + split[2]
    return end

class BucketPolicyAdder():
    def __init__(self, aws_region, vpc_id):

        self.aws_region = aws_region
        self.vpc_id = vpc_id

        self.short_region = convert_aws_region_to_short(self.aws_region)
        self.bucket_name = 'riano-deploy-%s' % self.short_region

        self.ec2 = boto3.resource('ec2', region_name=self.aws_region)
        self.s3 = boto3.resource('s3', region_name=self.aws_region)
        self.ec2_client = self.ec2.meta.client

        print(self.bucket_name)

        self.bucket_policy = self.s3.BucketPolicy(self.bucket_name)
        try:
            self.bucket_policy_json = json.loads(self.bucket_policy.policy)
        except ClientError as e:
            if e.response["Error"]["Code"] == "NoSuchBucketPolicy":
                print("bucket policy doesnt exist -- creating")
                self.bucket_policy_json = {
                          "Version": "2012-10-17",
                          "Id": self.bucket_name+"-policy",
                          "Statement": []
                        }
                return
            raise e


    def get_vpc_endpoint(self, vpc_id):
        vpces = self.ec2_client.describe_vpc_endpoints(
            Filters=[
                {
                    'Name': 'vpc-id',
                    'Values': [
                        vpc_id
                        ]
                    }
                ]
            )['VpcEndpoints']

        print(vpces)

        if len(vpces) == 0:
            return None

        return vpces[0]


    def add_vpc_to_policy(self):
        policy_json = self.bucket_policy_json
        vpc_endpoint = self.get_vpc_endpoint(self.vpc_id)

        if vpc_endpoint is not None:
            vpc_endpoint_id = vpc_endpoint['VpcEndpointId']
        else:
            print("ERROR: no vpc_endpoint -- exiting")
            sys.exit(1)

        new_statement = {
            'Sid': self.vpc_id,
            'Effect': 'Allow',
            'Principal': '*',
            'Action': [
                's3:ListBucket',
                's3:GetObject'
                ],
            'Resource': [
                'arn:aws:s3:::%s/*' % self.bucket_name,
                'arn:aws:s3:::%s' % self.bucket_name
                ],
            'Condition': {
                'StringEquals': {
                    'aws:sourceVpc': self.vpc_id
                    }
                }
            }

        if vpc_endpoint_id is not None:
            new_statement['Condition']['StringEquals']['aws:sourceVpce'] = vpc_endpoint_id

        policy_json['Statement'].append(new_statement)

        new_policy = json.dumps(policy_json, sort_keys=True)
        print("PUTing S3 policy to vpc %s: %s" % (self.vpc_id, new_policy))
        self.bucket_policy.put(
            ConfirmRemoveSelfBucketAccess=False,
            Policy=new_policy
            )

if __name__ == "__main__":
    ARGS = PARSER.parse_args()
    AWS_REGION = ARGS.aws_region
    VPC_ID = ARGS.vpc_id
    BucketPolicyAdder(AWS_REGION, VPC_ID).add_vpc_to_policy()
