import json
import ipaddress
import subprocess

from config import cfg

from tf_utils import get_newest_tagged_saltstack_commit, convert_aws_region_to_short
from python_terraform import Terraform, IsFlagged, IsNotFlagged, TerraformCommandError

class TerraformController:
    def __init__(self, aws_region, bebo_env, vpc_id, ec2_controller, files):
        self.aws_region = aws_region
        self.bebo_env = bebo_env
        self.vpc_id = vpc_id
        self.ec2_controller = ec2_controller
        self.files = files
        with open("./region_subnets.json", "r+") as vpc_subnets_file:
            self.region_subnets = json.loads(vpc_subnets_file.read())

        for _aws_region in self.region_subnets:
            str_id = self.region_subnets[_aws_region]
            self.region_subnets[_aws_region] = ipaddress.IPv4Address(str_id)

    def get_vpc_vals(self):
        vpc = {}
        ip = self.region_subnets.get(self.aws_region, None)
        if ip is not None:
            ip = ip.exploded
            vpc["cidr_block"] = ip + "/16"
            vpc["cidr_prefix"] = ip.split(".")[0] + "." + ip.split(".")[1]
        else:
            print("%s doesnt have a subnet listed -- exiting" % self.aws_region)
            raise KeyError("no subnet")

        return vpc

    def get_dc_vals(self):
        dc = {}
        dc["name"] = self.vpc_id
        dc["BEBO_ENV"] = self.bebo_env
        return dc

    def create_tfvars_data(self):
        bebo_domain = cfg.DOMAIN

        tfvars_data = {}
        tfvars_data["bebo_domain"] = bebo_domain
        tfvars_data["aws_domain"] = "aws." + bebo_domain
        tfvars_data["aws_region"] = self.aws_region
        tfvars_data["aws_short_region"] = convert_aws_region_to_short(self.aws_region)

        tfvars_data["ssl_cert"] = "bundle-key-star.%s.pem" % bebo_domain
        tfvars_data["nginx_cert"] = "bundle-star.%s.crt" % bebo_domain
        tfvars_data["nginx_key"] = "star.%s.key" % bebo_domain

        tfvars_data["vpc"] = self.get_vpc_vals()
        tfvars_data["dc"] = self.get_dc_vals()

        tfvars_data["key_name"] = "terraform-" + self.aws_region
        tfvars_data["key_filename"] = "/keybase/team/teambebo/keys/" + tfvars_data["key_name"] + ".pem"

        tfvars_data["core_ami"] = self.ec2_controller.get_core_ami()
        tfvars_data["composer_ami"] = self.ec2_controller.get_composer_ami()

        tfvars_data["zones"] = self.ec2_controller.get_aws_zones(self.aws_region)
        tfvars_data["zones_count"] = len(tfvars_data["zones"])

        tfvars_data["dns_zone_id"] = cfg.DNS_ZONE_ID

        tfvars_data["saltstack_hash"] = get_newest_tagged_saltstack_commit()

        if self.aws_region == "eu-west-3":
            tfvars_data["admin_size"] = "c5.large"
        else:
            tfvars_data["admin_size"] = "c4.xlarge"

        return tfvars_data

    def create(self, do_prompt, dry):
        cwd = "./vpcs/%s" % self.vpc_id

        command = "cp"
        for f in self.files:
            command = command + " ../../tf_src/%s" % f

        command = command + " ."

        print("Running command: \"%s\" to copy in tf files" % command)

        subprocess.Popen(command, stdout=subprocess.PIPE, shell=True, cwd=cwd)
        terraform = Terraform(working_dir=cwd)
        terraform.init(capture_output=dry)

        try:
            terraform.plan(state='terraform.tfstate', capture_output=False, raise_on_error=True, detailed_exitcode=IsNotFlagged)
        except TerraformCommandError as e:
            print("TerraformCommandError: %s" % e)
            return False

        if dry is True:
            try:
                subprocess.Popen('rm -rf %s' % cwd, stdout=subprocess.PIPE, shell=True, cwd=cwd+"/..")
            except Exception as e:
                print("failed to remove terraform vpc dir: %s" % e)
                return False

            return True

        if do_prompt is True:
            should_continue = input("Continue? \"yes\" or other only: \n") == "yes"
        else:
            should_continue = True

        if should_continue is False:
            return False

        try:
            terraform.plan(state='terraform.tfstate', out='terraform.tfplan', capture_output=False, raise_on_error=True, detailed_exitcode=IsNotFlagged)
            terraform.apply('terraform.tfplan', state_out='terraform.tfstate', capture_output=False, raise_on_error=True, detailed_exitcode=IsNotFlagged)
            return True
        except TerraformCommandError as e:
            print("TerraformCommandError: %s" % e)
            return False

    def destroy(self, do_prompt):
        terraform = Terraform(working_dir="./vpcs/%s" % self.vpc_id)
        terraform.init(capture_output=False)
        try:
            terraform.plan(destroy=IsFlagged, state='terraform.tfstate', capture_output=False, raise_on_error=True, detailed_exitcode=IsNotFlagged)
        except TerraformCommandError as e:
            print("TerraformCommandError: %s" % e)
            return False

        if do_prompt is True:
            should_continue = input("Continue? \"yes\" or other only: \n") == "yes"
        else:
            should_continue = True


        if should_continue is False:
            return False

        self.ec2_controller.terminate_extra_instances(self.vpc_id)

        try:
            terraform.plan(destroy=IsFlagged, state='terraform.tfstate', out='terraform.tfplan', capture_output=False, raise_on_error=True, detailed_exitcode=IsNotFlagged)
            terraform.apply('terraform.tfplan', state_out='terraform.tfstate', capture_output=False, raise_on_error=True, detailed_exitcode=IsNotFlagged)
            return True
        except TerraformCommandError as e:
            print("TerraformCommandError: %s" % e)
            return False

