# Composer Proxy Instances
resource "aws_instance" "composerproxy" {
  count = "${var.zones_count}"

  connection {
    # The default username for our AMI
    # user = "debian"
    user = "admin"

    private_key  = "${file(var.key_filename)}"
    bastion_host = "${aws_eip.admin.public_ip}"
    host         = "${self.private_ip}"
  }

  instance_type = "t2.medium"

  root_block_device {
    volume_size = "100"
  }

  instance_initiated_shutdown_behavior = "terminate"

  # Lookup the correct AMI based on the region
  # we specified
  ami = "${var.general_ami}"

  # The name of our SSH keypair we created above.
  key_name = "${var.key_name}"

  vpc_security_group_ids = ["${aws_default_security_group.default.id}",
    "${aws_security_group.public_https.id}",
  ]

  subnet_id = "${element(aws_subnet.subnet_public.*.id, count.index % var.zones_count)}"

  iam_instance_profile = "bebo-composerproxy"

  tags {
    Name = "${var.dc["name"]}-composerproxy-00${count.index}.${var.aws_domain}"
    Role = "composerproxy"
  }

  provisioner "file" {
    source      = "/keybase/team/teambebo/ssl/${var.ssl_cert}"
    destination = "${var.ssl_cert}"
  }

  provisioner "file" {
    destination = "install_salt.sh"
    content = "${data.template_file.salt_install.rendered}"
  }

  provisioner "remote-exec" {
    inline = [
      "until [ -f /var/lib/cloud/instance/boot-finished ]; do sleep 1 && echo sleep; done",
      "sudo rm -rf /srv/salt",
      "sudo sh -c 'echo ${var.dc["name"]}-composerproxy-00${count.index}.${var.aws_domain} > /etc/salt/minion_id'",
      "sudo sh -c 'echo \"master: ${aws_instance.admin.private_ip}\" > /etc/salt/minion.d/master.conf'",
      "sudo chmod +x /home/admin/install_salt.sh",
      "sudo /home/admin/install_salt.sh",
      "sudo mkdir -p /etc/haproxy",
      "sudo mv ${var.ssl_cert} /etc/haproxy/${var.ssl_cert}",
    ]
  }
}

resource "aws_eip" "composerproxy" {
  count    = "${var.zones_count}"
  instance = "${element(aws_instance.composerproxy.*.id, count.index)}"
  vpc      = true
}

# sgs
resource "aws_security_group" "public_https" {
  name        = "public_https"
  description = "public http and https"
  vpc_id      = "${aws_vpc.main_vpc.id}"

  ingress {
    protocol         = "tcp"
    from_port        = 80
    to_port          = 80
    cidr_blocks      = ["0.0.0.0/0"]
    ipv6_cidr_blocks = ["::/0"]
    description      = "public http"
  }

  ingress {
    protocol         = "tcp"
    from_port        = 443
    to_port          = 443
    cidr_blocks      = ["0.0.0.0/0"]
    ipv6_cidr_blocks = ["::/0"]
    description      = "public https"
  }

  tags {
    Name = "${aws_vpc.main_vpc.id}-public_https"
  }
}

# route53

resource "aws_route53_health_check" "composer-healthcheck" {
  type                   = "CALCULATED"
  child_health_threshold = 1
  child_healthchecks     = ["${element(aws_route53_health_check.composerproxy.*.id, count.index)}"]

  tags = {
    Name = "${var.dc["name"]}-composerhealth.${var.bebo_domain}"
  }
}

resource "aws_route53_health_check" "composerproxy" {
  count             = "${var.zones_count}"
  ip_address        = "${element(aws_eip.composerproxy.*.public_ip, count.index)}"
  port              = 443
  type              = "TCP"
  failure_threshold = "2"
  request_interval  = "10"
  measure_latency   = true

  tags = {
    Name = "${var.dc["name"]}-composerproxy-00${count.index}"
  }
}

resource "aws_route53_record" "obiwan" {
  count                            = "${var.zones_count}"
  zone_id                          = "${var.dns_zone_id}"
  name                             = "${var.dc["name"]}-obiwan.${var.bebo_domain}"
  type                             = "A"
  ttl                              = "10"
  health_check_id                  = "${element(aws_route53_health_check.composerproxy.*.id, count.index)}"
  multivalue_answer_routing_policy = true
  set_identifier                   = "${var.dc["name"]}-obiwan.00${count.index}"
  records                          = ["${element(aws_eip.composerproxy.*.public_ip, count.index)}"]
}

resource "aws_route53_record" "speedproxy" {
  count                            = "${var.zones_count}"
  zone_id                          = "${var.dns_zone_id}"
  name                             = "${var.dc["name"]}-speed.${var.bebo_domain}"
  type                             = "A"
  ttl                              = "10"
  health_check_id                  = "${element(aws_route53_health_check.composerproxy.*.id, count.index)}"
  multivalue_answer_routing_policy = true
  set_identifier                   = "${var.dc["name"]}-speed.00${count.index}"
  records                          = ["${element(aws_eip.composerproxy.*.public_ip, count.index)}"]
}

resource "aws_route53_record" "marioproxy" {
  count                            = "${var.zones_count}"
  zone_id                          = "${var.dns_zone_id}"
  name                             = "${var.dc["name"]}-mario.${var.bebo_domain}"
  type                             = "A"
  ttl                              = "10"
  health_check_id                  = "${element(aws_route53_health_check.composerproxy.*.id, count.index)}"
  multivalue_answer_routing_policy = true
  set_identifier                   = "${var.dc["name"]}-mario.00${count.index}"
  records                          = ["${element(aws_eip.composerproxy.*.public_ip, count.index)}"]
}


resource "aws_route53_health_check" "marioproxy" {
  count             = "${var.zones_count}"
  fqdn              = "mario.${var.bebo_domain}"
  ip_address        = "${element(aws_eip.composerproxy.*.public_ip, count.index)}"
  port              = 443
  type              = "HTTPS"
  resource_path     = "/"
  failure_threshold = "2"
  request_interval  = "10"
  measure_latency   = true

  tags = {
    Name = "${var.dc["name"]}-mario-00${count.index}"
  }
}


resource "aws_route53_health_check" "speedproxy" {
  count             = "${var.zones_count}"
  fqdn              = "speed.${var.bebo_domain}"
  ip_address        = "${element(aws_eip.composerproxy.*.public_ip, count.index)}"
  port              = 443
  type              = "HTTPS"
  resource_path     = "/"
  failure_threshold = "2"
  request_interval  = "10"
  measure_latency   = true

  tags = {
    Name = "${var.dc["name"]}-mario-00${count.index}"
  }
}

