# Create a VPC to launch our instances into
resource "aws_vpc" "main_vpc" {
  cidr_block           = "${var.vpc["cidr_block"]}"
  enable_dns_support   = true
  enable_dns_hostnames = true

  tags = {
    Name     = "${var.dc["name"]}"
    BEBO_ENV = "${var.dc["BEBO_ENV"]}"
  }
}

# dhcp options for the vpc
resource "aws_vpc_dhcp_options" "dhcp_opts" {
  domain_name         = "${var.aws_domain}"
  domain_name_servers = ["AmazonProvidedDNS"]

  tags {
    Name = "${var.dc["name"]}"
  }
}

resource "aws_vpc_dhcp_options_association" "vpc_dhcp_options_association" {
  vpc_id          = "${aws_vpc.main_vpc.id}"
  dhcp_options_id = "${aws_vpc_dhcp_options.dhcp_opts.id}"
}

# Create a way out to the internet
resource "aws_internet_gateway" "gw" {
  vpc_id = "${aws_vpc.main_vpc.id}"

  tags {
    Name = "InternetGateway"
  }
}

# Public route as way out to the internet
resource "aws_route" "internet_access" {
  route_table_id         = "${aws_vpc.main_vpc.main_route_table_id}"
  destination_cidr_block = "0.0.0.0/0"
  gateway_id             = "${aws_internet_gateway.gw.id}"
}

# Create the private route table
resource "aws_route_table" "private_route_table" {
  count  = "${var.zones_count}"
  vpc_id = "${aws_vpc.main_vpc.id}"

  tags {
    Name = "Private route table ${count.index}"
  }
}

# Create private route
resource "aws_route" "private_route" {
  count                  = "${var.zones_count}"
  route_table_id         = "${element(aws_route_table.private_route_table.*.id, count.index)}"
  destination_cidr_block = "0.0.0.0/0"
  nat_gateway_id         = "${element(aws_nat_gateway.nat-gateway.*.id, count.index)}"
}

resource "aws_subnet" "subnet_private" {
  count      = "${var.zones_count}"
  vpc_id     = "${aws_vpc.main_vpc.id}"
  cidr_block = "${var.vpc["cidr_prefix"]}.${count.index}.0/24"

  #map_public_ip_on_launch = true
  availability_zone = "${var.zones[count.index]}"

  tags = {
    Name = "${aws_vpc.main_vpc.id}-private-${count.index}"
  }
}

# public subnets
resource "aws_subnet" "subnet_public" {
  count                   = "${var.zones_count}"
  vpc_id                  = "${aws_vpc.main_vpc.id}"
  cidr_block              = "${var.vpc["cidr_prefix"]}.${10 + count.index * 4}.0/22"
  map_public_ip_on_launch = true
  availability_zone       = "${var.zones[count.index]}"

  tags = {
    Name = "${aws_vpc.main_vpc.id}-public-${count.index}"
  }
}

# Create an EIP for the natgateway
resource "aws_eip" "nat" {
  count      = "${var.zones_count}"
  vpc        = true
}

# Create a nat gateway and it will depend on the internet gateway creation
resource "aws_nat_gateway" "nat-gateway" {
  count         = "${var.zones_count}"
  allocation_id = "${element(aws_eip.nat.*.id, count.index)}"
  subnet_id     = "${element(aws_subnet.subnet_public.*.id, count.index)}"

  tags = {
    Name = "${aws_vpc.main_vpc.id}-public-${element(aws_subnet.subnet_public.*.id, count.index)}"
  }
}

# Associate subnet to public route table
resource "aws_route_table_association" "subnet_public-association" {
  count          = "${var.zones_count}"
  subnet_id      = "${element(aws_subnet.subnet_public.*.id, count.index)}"
  route_table_id = "${aws_vpc.main_vpc.main_route_table_id}"
}

# Associate subnets to private route table
resource "aws_route_table_association" "subnet_private_association" {
  count          = "${var.zones_count}"
  subnet_id      = "${element(aws_subnet.subnet_private.*.id, count.index)}"
  route_table_id = "${element(aws_route_table.private_route_table.*.id, count.index)}"
}

#VPCE

data "aws_vpc_endpoint_service" "s3" {
  service_name = "com.amazonaws.${var.aws_region}.s3"
}

resource "aws_vpc_endpoint" "vpc-s3-endpoint" {
  vpc_id          = "${aws_vpc.main_vpc.id}"
  service_name    = "${data.aws_vpc_endpoint_service.s3.service_name}"
  route_table_ids = ["${aws_route_table.private_route_table.*.id}", "${aws_vpc.main_vpc.main_route_table_id}"]
}
