# The provider here is aws but it can be other provider
provider "aws" {
  region = "${var.aws_region}"
}

# Create a VPC to launch our instances into
resource "aws_vpc" "composer" {
  cidr_block = "${var.vpc["cidr_block"]}"
  enable_dns_support = true
  enable_dns_hostnames = true
  tags = {
    Name = "${var.dc["name"]}"
    BEBO_ENV = "${var.dc["BEBO_ENV"]}"
  }
}

# dhcp options for the vpc
resource "aws_vpc_dhcp_options" "dhcp_opts" {
  domain_name          = "${var.aws_domain}"
  domain_name_servers  = ["AmazonProvidedDNS"]

  tags {
    Name = "${var.dc["name"]}"
  }
}

resource "aws_vpc_dhcp_options_association" "vpc_dhcp_options_association" {
  vpc_id          = "${aws_vpc.composer.id}"
  dhcp_options_id = "${aws_vpc_dhcp_options.dhcp_opts.id}"
}


# Create a way out to the internet
resource "aws_internet_gateway" "gw" {
  vpc_id = "${aws_vpc.composer.id}"
  tags {
    Name = "InternetGateway"
  }
}

# Public route as way out to the internet
resource "aws_route" "internet_access" {
  route_table_id = "${aws_vpc.composer.main_route_table_id}"
  destination_cidr_block = "0.0.0.0/0"
  gateway_id = "${aws_internet_gateway.gw.id}"
}


# Create the private route table
resource "aws_route_table" "private_route_table" {
  count = "${var.zones_count}"
  vpc_id = "${aws_vpc.composer.id}"

  tags {
    Name = "Private route table ${count.index}"
  }
}

# Create private route
resource "aws_route" "private_route" {
  count = "${var.zones_count}"
  route_table_id  = "${element(aws_route_table.private_route_table.*.id, count.index)}"
  destination_cidr_block = "0.0.0.0/0"
  nat_gateway_id = "${element(aws_nat_gateway.nat-gateway.*.id, count.index)}"
}

resource "aws_subnet" "subnet_private" {
  count = "${var.zones_count}"
  vpc_id = "${aws_vpc.composer.id}"
  cidr_block = "${var.vpc["cidr_prefix"]}.${count.index + 1}.0/24"
  #map_public_ip_on_launch = true
  availability_zone = "${var.zones[count.index]}"
  tags = {
    Name =  "${aws_vpc.composer.id}-private-${count.index}"
  }
}

# public subnets
resource "aws_subnet" "subnet_public" {
  count = "${var.zones_count}"
  vpc_id = "${aws_vpc.composer.id}"
  cidr_block = "${var.vpc["cidr_prefix"]}.${4 + count.index * 4}.0/22"
  map_public_ip_on_launch = true
  availability_zone = "${var.zones[count.index]}"
  tags = {
  Name =  "${aws_vpc.composer.id}-public-${count.index}"
  }
}

# Create an EIP for the natgateway
resource "aws_eip" "nat" {
  count = "${var.zones_count}"
  vpc = true
  depends_on = ["aws_internet_gateway.gw"]
}

# Create a nat gateway and it will depend on the internet gateway creation
resource "aws_nat_gateway" "nat-gateway" {
  count = "${var.zones_count}"
  allocation_id = "${element(aws_eip.nat.*.id, count.index)}"
  subnet_id = "${element(aws_subnet.subnet_public.*.id, count.index)}"
  depends_on = ["aws_internet_gateway.gw"]
  tags = {
    Name =  "${aws_vpc.composer.id}-public-${element(aws_subnet.subnet_public.*.id, count.index)}"
  }
}

# Associate subnet to public route table
resource "aws_route_table_association" "subnet_public-association" {
  count = "${var.zones_count}"
  subnet_id = "${element(aws_subnet.subnet_public.*.id, count.index)}"
  route_table_id = "${aws_vpc.composer.main_route_table_id}"
}

# Associate subnets to private route table
resource "aws_route_table_association" "subnet_private_association" {
  count = "${var.zones_count}"
  subnet_id = "${element(aws_subnet.subnet_private.*.id, count.index)}"
  route_table_id = "${element(aws_route_table.private_route_table.*.id, count.index)}"
}


# Admin Server Intance

resource "aws_instance" "admin" {

  # The connection block tells our provisioner how to
  # communicate with the resource (instance)
  connection {
    # The default username for our AMI
    # user = "debian"
    user = "admin"
    private_key = "${file(var.key_filename)}"

    # The connection will use the local SSH agent for authentication.
  }
  instance_type = "m4.xlarge"

  root_block_device {
    volume_size = "100"
  }

  # Lookup the correct AMI based on the region
  # we specified
  ami = "${var.general_ami}"

  # The name of our SSH keypair we created above.
  key_name = "${var.key_name}"

  # Our Security group to allow HTTP and SSH access
  vpc_security_group_ids = ["${aws_default_security_group.default.id}",
                            "${aws_security_group.admin.id}"]

  # We're going to launch into the same subnet as our ELB. In a production
  # environment it's more common to have a separate private subnet for
  # backend instances.
  subnet_id = "${aws_subnet.subnet_public.0.id}"

  iam_instance_profile = "bebo-admin"

  tags {
    Name = "${var.dc["name"]}-admin-000.${var.aws_domain}"
  }

  provisioner "file" {
    source = "/keybase/team/teambebo/keys/admin.rsa"
    destination = "/home/admin/.ssh/id_rsa"
  }

  provisioner "file" {
    source = "/keybase/team/teambebo/keys/admin.rsa"
    destination = "/home/admin/.ssh/admin.aws"
  }

  provisioner "file" {
    source = "/keybase/team/teambebo/keys/known_hosts"
    destination = "/home/admin/.ssh/known_hosts"
  }

  provisioner "file" {
    source = "./aws-saltstack/consul/files/consul.json.bootstrap"
    destination = "consul.json"
  }

  provisioner "file" {
    source = "/keybase/team/teambebo/ssl/${var.ssl_cert}"
    destination = "${var.ssl_cert}"
  }

  provisioner "file" {
    source = "/keybase/team/teambebo/ssl/${var.nginx_cert}"
    destination = "${var.nginx_cert}"
  }

  provisioner "file" {
    source = "/keybase/team/teambebo/ssl/${var.nginx_key}"
    destination = "${var.nginx_key}"
  }

  provisioner "file" {
    source = "${var.key_filename}"
    destination = "${var.key_name}"
  }

  provisioner "file" {
    source = "files/pillar_top.tpl"
    destination = "/tmp/pillar_top.sls"
  }

  provisioner "file" {
    content = "consul:\n  datacenter: ${var.dc["name"]}\nrsyslog:\n  logserver: ${aws_instance.admin.private_ip}\n  port: 514"
    destination = "/tmp/pillar_base_init.sls"
  }

  provisioner "remote-exec" {
    inline = [
      "until [ -f /var/lib/cloud/instance/boot-finished ]; do sleep 1 && echo sleep; done",
      "sudo chown admin /home/admin/.ssh/id_rsa",
      "sudo chown admin /home/admin/.ssh/admin.aws",
      "sudo chown admin /home/admin/.ssh/known_hosts",
      "sudo chown 0600 /home/admin/.ssh/id_rsa",
      "sudo chown 0600 /home/admin/.ssh/admin.aws",
      "sudo chown 0600 /home/admin/.ssh/known_hosts",
      "sudo mkdir -p /srv/pillar/base",
      "sudo cp /tmp/pillar_top.sls /srv/pillar/top.sls",
      "sudo cp /tmp/pillar_base_init.sls /srv/pillar/base/init.sls",
      "sudo rm -rf /srv/salt",
      "sudo chown admin /srv",
      "git clone git@github.com:bebo/aws-saltstack.git /srv/salt",
      "sudo chown root /srv",
      "sudo mkdir -p /etc/salt/minion.d",
      "sudo sh -c 'echo \"master: ${aws_instance.admin.private_ip}\" > /etc/salt/minion.d/master.conf'",
      "sudo sh -c 'echo ${var.dc["name"]}-admin-000.${var.aws_domain} > /etc/salt/minion_id'",
      "sudo mkdir -p /srv/pillar/base",
      "sudo apt install -y salt-master salt-minion",
      "sudo salt-call --local state.highstate --file-root=/srv/salt --pillar-root=/srv/pillar --retcode-passthrough -l quiet",
      "sudo salt-call --local state.sls saltmaster --file-root=/srv/salt --pillar-root=/srv/pillar --retcode-passthrough -l quiet",
      "sudo salt-key -a ${var.dc["name"]}-admin-000.${var.aws_domain} -y",
      # setup consul
      #"sudo apt install -y jq moreutils || sleep 30 && sudo apt install -y jq moreutils",
      #"cat consul.json | jq '.datacenter = \"${var.dc["name"]}\"' | sponge consul.json",
      #"sudo cp consul.json /etc/consul/consul.json",
      #"sudo systemctl restart consul.service",
      # TODO: move to vault
      "sudo mkdir -p /srv/secure-files/ssl",
      "sudo cp ${var.ssl_cert} /srv/secure-files/ssl/",
      "sudo cp ${var.nginx_cert} /srv/secure-files/ssl/",
      "sudo cp ${var.nginx_key} /srv/secure-files/ssl/",
      "sudo chown -R root:root /srv/secure-files",
      "sudo chmod 700 -R /srv/secure-files/",
      "sudo chmod 600 /srv/secure-files/ssl/${var.ssl_cert}",
      "sudo chmod 600 /srv/secure-files/ssl/${var.nginx_cert}",
      "sudo chmod 600 /srv/secure-files/ssl/${var.nginx_key}",
      "sudo mkdir -p /etc/salt/ssl",
      "sudo cp ${var.key_name} /etc/salt/ssl/${var.key_name}_id_rsa",
      "sudo chmod 0400 /etc/salt/ssl/${var.key_name}_id_rsa",
    ]
  }
    # https://github.com/hashicorp/terraform/issues/16678
    # provisioner "salt-masterless" {
    #    local_state_tree = "./aws-saltstack"
    #    bootstrap_args = "-M -L -i ${var.dc["name"]}-admin-000.${var.aws_domain}"
    # }
}


resource "aws_ebs_volume" "admin_syslog" {
  availability_zone = "${aws_instance.admin.availability_zone}"
  size              = 250
  tags {
    Name =  "${aws_vpc.composer.id}-admin_syslog"
  }
}

resource "aws_volume_attachment" "admin_syslog" {
  device_name = "/dev/xvdb"
  volume_id   = "${aws_ebs_volume.admin_syslog.id}"
  instance_id = "${aws_instance.admin.id}"
}

resource "aws_eip" "admin" {
  vpc = true
  instance = "${aws_instance.admin.id}"
}


# Consul Instances

resource "aws_instance" "consul" {

  count = 3  # no max/min and we can either do 3 or 5

  connection {
    # The default username for our AMI
    # user = "debian"
    user = "admin"
    private_key = "${file(var.key_filename)}"
    bastion_host = "${aws_eip.admin.public_ip}"
  }
  instance_type = "c4.large"

  root_block_device {
    volume_size = "100"
  }

  # Lookup the correct AMI based on the region
  # we specified
  ami = "${var.general_ami}"

  # The name of our SSH keypair we created above.
  key_name = "${var.key_name}"

  vpc_security_group_ids = ["${aws_default_security_group.default.id}"]

  subnet_id = "${element(aws_subnet.subnet_private.*.id, count.index % var.zones_count)}"

  iam_instance_profile = "bebo-consul"

  tags {
    Name = "${var.dc["name"]}-consul-00${count.index}.${var.aws_domain}"
    Role = "consul"
  }

  provisioner "file" {
    source = "./aws-saltstack/consul/files/consul-limits.service"
    destination = "consul.limits.conf"
  }

  provisioner "file" {
    source = "./aws-saltstack/consul/files/consul.json.bootstrap"
    destination = "consul.json"
  }

  provisioner "file" {
    source = "./aws-saltstack/consul/files/server.conf"
    destination = "consul.server.json"
  }

  provisioner "file" {
    source = "./aws-saltstack/consul/files/consul.service"
    destination = "consul.service"
  }

  provisioner "remote-exec" {
    inline = [
      #"sudo apt install -y jq moreutils || sleep 30 && sudo apt install -y jq moreutils",
      #"cat consul.json | jq '.datacenter = \"${var.dc["name"]}\"' | sponge consul.json",
      #"sudo cp consul.json /etc/consul/consul.json",
      #"sudo mkdir -p /etc/systemd/system/consul.service.d",
      #"sudo cp consul.limits.conf /etc/systemd/system/consul.service.d/limits.conf",
      #"sudo cp consul.server.json /etc/consul/conf.d/server.json",
      #"sudo systemctl restart consul.service",
      "until [ -f /var/lib/cloud/instance/boot-finished ]; do sleep 1 && echo sleep; done",
      "sudo rm -rf /srv/salt",
      "sudo sh -c 'echo ${var.dc["name"]}-consul-00${count.index}.${var.aws_domain} > /etc/salt/minion_id'",
      "sudo mkdir -p /etc/salt/minion.d",
      "sudo sh -c 'echo \"master: ${aws_instance.admin.private_ip}\" > /etc/salt/minion.d/master.conf'",
      "sudo apt install -y salt-minion",
     # give consul some time to come up:
      "sleep 10s",
      #"systemctl daemon-reload",
      #"systemctl restart consul",
      #"/usr/share/consul/bin/consul kv put salt-shared/consul/datacenter ${var.dc["name"]} || true",
      #"/usr/share/consul/bin/consul kv put salt-shared/rsyslog/logserver ${self.private_ip} || true",
      #"/usr/share/consul/bin/consul kv put salt-shared/rsyslog/port 514 || true"
    ]
  }

  depends_on = ["aws_instance.admin", "aws_route.private_route", "aws_eip.admin"]
}

# Composer Proxy Instances
resource "aws_instance" "composerproxy" {

  count = "${var.zones_count}"

  connection {
    # The default username for our AMI
    # user = "debian"
    user = "admin"
    private_key = "${file(var.key_filename)}"
    bastion_host = "${aws_eip.admin.public_ip}"
    host = "${self.private_ip}"
  }
  instance_type = "t2.medium"

  root_block_device {
    volume_size = "100"
  }

  # Lookup the correct AMI based on the region
  # we specified
  ami = "${var.general_ami}"

  # The name of our SSH keypair we created above.
  key_name = "${var.key_name}"

  vpc_security_group_ids = ["${aws_default_security_group.default.id}",
                                                        "${aws_security_group.public_https.id}"]

  subnet_id = "${element(aws_subnet.subnet_public.*.id, count.index % var.zones_count)}"

  iam_instance_profile = "bebo-composerproxy"

  tags {
    Name = "${var.dc["name"]}-composerproxy-00${count.index}.${var.aws_domain}"
    Role = "composerproxy"
  }

  provisioner "file" {
    source = "./aws-saltstack/consul/files/consul.json.bootstrap"
    destination = "consul.json"
  }

#  provisioner "file" {
#    source = "./aws-saltstack/consul/files/consul.service"
#    destination = "consul.service"
#  }

  provisioner "remote-exec" {
    inline = [
      "until [ -f /var/lib/cloud/instance/boot-finished ]; do sleep 1 && echo sleep; done",
      #"sudo apt install -y jq moreutils || sleep 30 && sudo apt install -y jq moreutils",
      #"cat consul.json | jq '.datacenter = \"${var.dc["name"]}\"' | sponge consul.json",
      #"sudo cp consul.json /etc/consul/consul.json",
      #"sudo systemctl restart consul.service",
      "sudo rm -rf /srv/salt",
      "sudo sh -c 'echo ${var.dc["name"]}-composerproxy-00${count.index}.${var.aws_domain} > /etc/salt/minion_id'",
      "sudo sh -c 'echo \"master: ${aws_instance.admin.private_ip}\" > /etc/salt/minion.d/master.conf'",
      "sudo apt install -y salt-minion"
    ]
  }

  depends_on = ["aws_instance.admin", "aws_instance.consul", "aws_eip.admin"]
}

resource "aws_eip" "composerproxy" {
    count = "${var.zones_count}"
    instance = "${element(aws_instance.composerproxy.*.id, count.index)}"
    vpc      = true
}
