# Consul Instances

resource "aws_instance" "consul" {

  count = 3  # no max/min and we can either do 3 or 5

  connection {
    # The default username for our AMI
    user = "admin"
    private_key = "${file(var.key_filename)}"
    bastion_host = "${aws_eip.admin.public_ip}"
  }

  instance_type = "c4.large"

  root_block_device {
    volume_size = "100"
  }

  instance_initiated_shutdown_behavior = "terminate"

  # Lookup the correct AMI based on the region
  # we specified
  ami = "${var.general_ami}"

  # The name of our SSH keypair we created above.
  key_name = "${var.key_name}"

  vpc_security_group_ids = ["${aws_default_security_group.default.id}"]

  subnet_id = "${element(aws_subnet.subnet_private.*.id, count.index % var.zones_count)}"

  iam_instance_profile = "bebo-consul"

  tags {
    Name = "${var.dc["name"]}-consul-00${count.index}.${var.aws_domain}"
    Role = "consul"
  }

  provisioner "remote-exec" {
    inline = [
      "until [ -f /var/lib/cloud/instance/boot-finished ]; do sleep 1 && echo sleep; done",
      "sudo rm -rf /srv/salt",
      "sudo sh -c 'echo ${var.dc["name"]}-consul-00${count.index}.${var.aws_domain} > /etc/salt/minion_id'",
      "sudo mkdir -p /etc/salt/minion.d",
      "sudo sh -c 'echo \"master: ${aws_instance.admin.private_ip}\" > /etc/salt/minion.d/master.conf'",
      "wget -O - https://repo.saltstack.com/apt/debian/9/amd64/archive/2017.7.4/SALTSTACK-GPG-KEY.pub | sudo apt-key add - && sudo touch /etc/apt/sources.list.d/saltstack_debian_repo.list && sudo chmod 0666 /etc/apt/sources.list.d/saltstack_debian_repo.list && echo \"deb http://repo.saltstack.com/apt/debian/9/amd64/archive/2017.7.4 stretch main\" > /etc/apt/sources.list.d/saltstack_debian_repo.list && sudo chmod 0644 /etc/apt/sources.list.d/saltstack_debian_repo.list && sudo apt update && sudo apt install -y salt-minion || sudo fuser /var/lib/apt/lists/lock | awk -F: '{ print $1 }' | sudo xargs kill -15 && sudo apt install -y salt-minion"
    ]
  }

  depends_on = ["aws_instance.admin", "aws_route.private_route", "aws_eip.admin"]
}
