# Consul Instances

resource "aws_instance" "consul" {
  count = 3 # no max/min and we can either do 3 or 5

  connection {
    # The default username for our AMI
    user         = "admin"
    private_key  = "${file(var.key_filename)}"
    bastion_host = "${aws_eip.admin.public_ip}"
  }

  instance_type = "c4.large"

  root_block_device {
    volume_size = "100"
  }

  instance_initiated_shutdown_behavior = "terminate"

  # Lookup the correct AMI based on the region
  # we specified
  ami = "${var.general_ami}"

  # The name of our SSH keypair we created above.
  key_name = "${var.key_name}"

  vpc_security_group_ids = ["${aws_default_security_group.default.id}"]

  subnet_id = "${element(aws_subnet.subnet_private.*.id, count.index % var.zones_count)}"

  iam_instance_profile = "bebo-consul"

  tags {
    Name = "${var.dc["name"]}-consul-00${count.index}.${var.aws_domain}"
    Role = "consul"
  }

  provisioner "file" {
    destination = "install_salt.sh"
    content = "${data.template_file.install_salt.rendered}"
  }

  provisioner "remote-exec" {
    inline = [
      "until [ -f /var/lib/cloud/instance/boot-finished ]; do sleep 1 && echo sleep; done",
      "sudo rm -rf /srv/salt",
      "sudo sh -c 'echo ${var.dc["name"]}-consul-00${count.index}.${var.aws_domain} > /etc/salt/minion_id'",
      "sudo mkdir -p /etc/salt/minion.d",

      "sudo sh -c 'echo \"master: ${aws_instance.admin.private_ip}\" > /etc/salt/minion.d/master.conf'",
      "sudo chmod +x /home/admin/install_salt.sh",
      "sudo /home/admin/install_salt.sh"
    ]
  }

}

