
import numpy as np
from absl.flags import FLAGS
from absl import app, flags, logging
import cv2
import streamlink
import time
import sys
import random

import gi
gi.require_version('Gst', '1.0')
from gi.repository import Gst, GObject, GLib

flags.DEFINE_string("twitch", None, "")
flags.DEFINE_bool('gpu', False, "")


def get_twitch_url(channel_name):
    stream = streamlink.streams('twitch.tv/' + channel_name)
    logging.info(stream.keys())
    url = stream.get('720p', stream.get('1080p')).url
    return url


def link_many(*args):
    if not args:
        return
    for index, element in enumerate(args[:-1]):
        next_element = args[index+1]
        if not Gst.Element.link(element, next_element):
            raise TypeError('failed to link {}, {}'.format(
                element, next_element))


def make_queue(queue_name="", **kwargs):
    props = {k.replace("_", "-"): v for k, v in kwargs.items()}
    if queue_name:
        queue_name += "_"
    name = "%s_queue" % (queue_name)
    queue = Gst.ElementFactory.make("queue", name)
    queue.set_property("max-size-bytes", props.pop("max-size-bytes", 0))
    queue.set_property("max-size-buffers", props.pop("max-size-buffers", 0))
    queue.set_property("max-size-time", props.pop("max-size-time", 0))
    queue.set_property("silent", props.pop("silent", True))
    for k, v in props.items():
        queue.set_property(k, v)
    return queue


def make(element_name, name=None, **kwargs):
    props = {k.replace("_", "-"): v for k, v in kwargs.items()}
    element = Gst.ElementFactory.make(element_name, name)
    if not element:
        raise TypeError('Failed to create element: {}, {} {}'.format(
            element_name, name, props))
    for k, v in props.items():
        element.set_property(k, v)
    return element


def make_caps(caps_str):
    return make("capsfilter", caps=Gst.Caps.from_string(caps_str))


class Pipeline:
    def __init__(self, source):
        self.pipeline = None
        self.source = source
        self.klt = 0

        self.timestamp = 0

    def fakesink_handoff(self, _, buf, pad):
        try:
            caps = pad.get_current_caps().to_string()
            now = self.pipeline.get_pipeline_clock().get_time()
            self.lt = now

            caps_structure = pad.get_current_caps().get_structure(0)
            width = caps_structure.get_int("width").value
            height = caps_structure.get_int("height").value

            timestamp = buf.pts
            (_result, mapinfo) = buf.map(Gst.MapFlags.READ)
            frame = np.frombuffer(
                mapinfo.data, np.uint8).reshape(height, width, 3)
            buf.unmap(mapinfo)

            delta = timestamp - self.timestamp
            self.timestamp = timestamp

            logging.info(f"ts: {timestamp}, delta: {delta / 1e9}")

            cv2.imwrite('test.jpg', cv2.cvtColor(frame, cv2.COLOR_RGB2BGR))
        except Exception as e:
            logging.exception("fakesink_handoff")

    def parser_buffer_probe_cb(self, pad, info):
        buf = info.get_buffer()
        if buf.has_flags(Gst.BufferFlags.DELTA_UNIT):  # Not keyframe
            return Gst.PadProbeReturn.DROP

        now = self.pipeline.get_pipeline_clock().get_time()
        # print("pipeline", "time:",
        #       (now - self.klt) / 1e6, "KEYFRAME FOUND", buf)
        self.klt = now

        return Gst.PadProbeReturn.PASS

    def on_tsdemux_cb(self, element, pad):
        caps = pad.get_current_caps().to_string()
        logging.info(f"on_ts_demux caps: {caps}")

        if not "video" in caps:
            return

        parser = make("h264parse")
        parser.get_static_pad("src").add_probe(
            Gst.PadProbeType.BUFFER, self.parser_buffer_probe_cb)

        decode_queue_pre = make_queue(
            "decoder-pre", max_size_buffers=5, leaky=1)
        decoder = make("nvdec") if FLAGS.gpu else make("avdec_h264")
        decode_queue_post = make_queue(
            "decoder-post", max_size_buffers=5, leaky=1)
        caps_filter = make_caps("video/x-raw,format=RGB")
        videoconvert = make("videoconvert")
        sink = make("fakesink", signal_handoffs=True)
        sink.connect("handoff", self.fakesink_handoff)

        elements = [parser, decode_queue_pre, decoder,
                    decode_queue_post, videoconvert, caps_filter, sink]

        for s in elements:
            self.pipeline.add(s)

        pad.link(elements[0].get_static_pad('sink'))
        link_many(*elements)

        for e in elements:
            e.set_state(Gst.State.PLAYING)

    def on_hlsdemux_cb(self, element, pad):
        caps = pad.get_current_caps().to_string()
        logging.info(f"on_hls_demux caps: {caps}")

        if not "mpegts" in caps:
            return

        ts_demuxer = make("tsdemux")
        self.pipeline.add(ts_demuxer)
        ts_demuxer.connect("pad-added", self.on_tsdemux_cb)
        pad.link(ts_demuxer.get_static_pad("sink"))
        ts_demuxer.set_state(Gst.State.PLAYING)

    def start(self):
        self.pipeline = Gst.Pipeline.new("pipeline")

        uri = get_twitch_url(self.source)

        src = make("souphttpsrc", location=uri)
        demux = make("hlsdemux")
        demux.connect("pad-added", self.on_hlsdemux_cb)

        elements = [src, demux]

        for s in elements:
            self.pipeline.add(s)

        link_many(*elements)

        self.pipeline.set_state(Gst.State.PLAYING)


def main(_):
    Gst.init(None)
    p = Pipeline(FLAGS.twitch)
    p.start()

    loop = GLib.MainLoop()
    loop.run()


if __name__ == '__main__':
    app.run(main)
