import logging
import pathlib
import shutil
import sys
import signal
import multiprocessing
import torch

import gi
import websocket
import sentry_sdk
gi.require_version("Gst", "1.0")
from gi.repository import Gst, GLib
Gst.init(None)

from bebo.utils import init_sentry
from bran.controllers.service_health_controller import ServiceHealthController
from bran.controllers.session_controller import SessionController
from bran.controllers.tracker_controller import TrackerController
from bran.api.sqs import SQSSingleton
from bran.models import download_models
from bran.shared.config import config
from bran.shared.logger import log

logging.getLogger(
    'requests.packages.urllib3.connectionpool').setLevel(logging.ERROR)
logging.getLogger("boto3").setLevel(logging.ERROR)
logging.getLogger("botocore").setLevel(logging.ERROR)
logging.getLogger("nose").setLevel(logging.ERROR)
logging.getLogger("s3transfer").setLevel(logging.ERROR)

init_sentry(dsn=config.SENTRY_URL)
torch.set_num_threads(1)

def ensure_directories():
    try:
        shutil.rmtree(config.TEMP_IMAGE_DIR)
    except FileNotFoundError as e:
        log.warning("Removing temp image directory failed: {}".format(e))

    if config.ENV == "local":
        try:
            log.info("local -- removing directories")
            shutil.rmtree("%s/%s" %
                          (config.DISPATCHER_BASE_DIR, "gamesense_media"))
            shutil.rmtree("%s/%s" %
                          (config.DISPATCHER_BASE_DIR, "gamesense_dot"))
            shutil.rmtree("/var/bran/gamesense_media")
        except Exception:
            sentry_sdk.capture_exception()
            log.error("Unable to remove directories")

    directories = ["/var/bran/recordings", "{}/gamesense_media".format(config.DISPATCHER_BASE_DIR), "{}/gamesense_dot".format(
        config.DISPATCHER_BASE_DIR), "/var/bran/gamesense_media", config.TEMP_IMAGE_DIR]
    for d in directories:
        log.info("ensuring directory: {}".format(d))
        pathlib.Path(d).mkdir(parents=True, exist_ok=True)

if __name__ == '__main__':
    log.info("Bran Version -- {} {}".format(config.VERSION, config.ENV))
    log.info("Gst Version -- %s", Gst.version_string())

    ensure_directories()
    try:
        tracker_controller = TrackerController()
        tracker_controller.start()
        download_models()

        session_controller = SessionController()
        health_controller = ServiceHealthController(
            session_controller=session_controller)
        health_controller.start()
        SQSSingleton.start()

        loop = GLib.MainLoop()
        loop.run()
    except KeyboardInterrupt:
        log.info("stopping")
    except Exception as e:
        sentry_sdk.capture_exception()
        log.error("Uncaught exception: {}".format(e), exc_info=True)
        sys.exit(1)
