import multiprocessing
import uuid
import time
import queue

import cv2
import numpy as np
import sentry_sdk

from bran.api.pharah import PharahAPI
from bran.detectors.worker import DetectorWorker
from bran.shared.logger import log
import bran.shared.util as util
import bran.shared.image_utils as ImageUtils
from bran.shared.config import config

MAX_QUEUE_SIZE = 144


class DetectorController:
    def __init__(self, **kwargs):
        self.detectors = []
        self.session = kwargs["session"]
        self.name = kwargs["name"]

        self.running = False
        self.queue = multiprocessing.Queue(MAX_QUEUE_SIZE)
        self.detector_worker = DetectorWorker(self.queue, self)
        self.detector_worker.daemon = True

        self.new_frame_track_index = -1
        self.new_frame_state_index = -1

    def stop(self):
        self.running = False
        self.queue.put('STOP')
        self.detector_worker.join()
        log.info("{} Detector controller stopped".format(self.name))

    def start(self):
        self.running = True
        self.detector_worker.start()

    def new_frame(self, game_frame, timestamp):
        if not self.running:
            return
        user_id = self.session.user_id
        log.debug("{} new_frame ts: {} {}".format(
            self.name, timestamp, game_frame.shape))

        self.new_frame_track_index += 1
        if self.new_frame_track_index % 30 == 0:
            util.track_event({
                "label": "new_frame",
                "timestamp": timestamp,
                "user_id": user_id,
            })

        frame_id = uuid.uuid4().hex

        try:
            # log.info("queue put ts: {}, frame_id: {}, user_game_id: {}".format(timestamp, frame_id, self.session.user_game_id))
            self.queue.put_nowait(
                (game_frame, timestamp, frame_id, self.session.user_game_id))
        except queue.Full:
            log.warning("{} Detector queue size too large, {}".format(
                self.name,
                self.queue.qsize()))
            util.track_event({
                "label": "error",
                "error": "Detector queue size too large",
                "user_id": user_id,
                "queue_size": self.queue.qsize()
            })
            self.session.on_timeout(user_id)
            return
        except:
            log.exception("failed to message to queue")

        self.new_frame_state_index += 1
        if self.new_frame_state_index % 10 == 0:
            if np.sum(game_frame) < 1:
                util.track_event({
                    "label": "error",
                    "error": "Frame is black",
                    "user_id": user_id,
                    "queue_size": self.queue.qsize()
                })
                log.warning("{} Frame is black".format(self.name))
                ImageUtils.save_preclassify_image(
                    game_frame, frame_id, 'Fortnite', 'black_frame', 'yes')
            else:
                PharahAPI.put_user_state(user_id, {
                    'exist': True,
                    'game': 'fortnite',
                    'mode': 'duo',
                    'state': 'lobby'
                })
