from queue import Queue
from threading import Thread, Lock
import math
import uuid

import sentry_sdk
import redis_lock

from bran.api.redis import RegionalRedis
from bran.api.sqs import SQSSingleton
from bran.api.pharah import PharahAPI
from bran.session import Session
from bran.shared.user import UserAPI
from bran.shared.config import config
from bran.shared.logger import log
import bran.shared.util as util


class SessionController(object):
    def __init__(self, **kwargs):

        self.redis = RegionalRedis(config.REDIS_LOCK_DB).client
        self.session_map = {}
        self.lock_map = {}
        self.lock_map_lock = Lock()
        SQSSingleton.register_handler(self.handle_ws_payload)

    def get_or_create_lock(self, user_id):
        with self.lock_map_lock:
            if user_id not in self.lock_map:
                self.lock_map[user_id] = redis_lock.Lock(
                    self.redis, user_id, expire=config.REDIS_LOCK_TTL, id=config.HOSTNAME.encode(), auto_renewal=True)
        return self.lock_map[user_id]

    def create_session(self, user_id, relay_url=None, mixer_url=None):
        log.debug("create session: user_id: {}, relay_url: {}, mixer_url: {}".format(
            user_id, relay_url, mixer_url))
        lock_track_payload = {}
        lock_track_payload["user_id"] = user_id
        lock_track_payload["relay_url"] = relay_url
        lock_track_payload["mixer_url"] = mixer_url
        name = '[{}]'.format(user_id)

        lock = self.get_or_create_lock(user_id)
        if lock._held:
            log.info("{} Holding the lock, nothing to do".format(name))

            if user_id not in self.session_map:
                if lock._lock_renewal_thread is None:
                    log.info(
                        "{} Newly created lock is already held, might due to bran restart".format(name))
                    self.delete_session(user_id)
                else:
                    log.error(
                        "{} This is wrong!!!! Zombie lock renewal thread".format(name))
                    util.track_event({
                        "label": "error",
                        "error": "session not in session map",
                        "user_id": user_id,
                    })
                    self.delete_session(user_id)

            lock_track_payload["label"] = "lock.held"
            util.track_event(lock_track_payload)
            return

        if self.get_health()["full"]:
            log.info("Box full, don't accept new stream")

            lock_track_payload["label"] = "lock.full"
            util.track_event(lock_track_payload)
            return

        try:
            if lock.acquire(blocking=False):
                log.info("{} Lock acquired, creating session with relay_url: {}, mixer_url {}".format(
                    name,
                    relay_url,
                    mixer_url))

                user = UserAPI.get_user_from_api(user_id)
                match_id = user.get('active_match_id', None)
                if not match_id:
                    log.warn("{} no match id, wtf???".format(
                        user.get('username', None)))
                    match = {'match_id': 'fake', 'game_id': 'fortnite'}
                else:
                    match = PharahAPI.get_match(match_id, user_id)[0]

                session = Session(
                    user_id, match, relay_url=relay_url, mixer_url=mixer_url)
                session.on_timeout = self.on_session_timeout
                session.start()
                self.session_map[user_id] = session

                lock_track_payload["label"] = "lock.acquired"
                util.track_event(lock_track_payload)
            else:
                log.debug("{} Lock exists, nothing to do".format(name))

                if user_id in self.session_map:
                    log.error(
                        "Why the f is session {} in session_map".format(user_id))
                    util.track_event({
                        "label": "error",
                        "error": "session in session map",
                        "user_id": user_id,
                    })
                    self.delete_session(user_id)

                lock_track_payload["label"] = "lock.exists"
                util.track_event(lock_track_payload)
        except Exception:
            log.exception(
                "{} Unable to create session fuck me".format(name))
            sentry_sdk.capture_exception()
            self.delete_session(user_id)
            util.track_event({
                "label": "error",
                "error": "Unable to create sesssion",
                "user_id": user_id,
            })

    def on_user_game(self, payload):
        try:
            log.info("on_user_game {}".format(payload))
            user_game = payload.get('result', [{}])[0]
            user_id = user_game.get('user_id', None)
            if user_id in self.session_map:
                log.info("user_is in self.session_map")
                ended_dttm = user_game.get('ended_dttm', None)
                user_game_id = None
                if not ended_dttm:
                    user_game_id = user_game.get('user_game_id', None)
                self.session_map[user_id].on_user_game(user_game_id)
        except:
            log.exception("on_user_game failed")

    def handle_ws_payload(self, payload):
        url = payload["url"]
        log.info("handle_ws_payload <<< {}".format(url))
        try:
            user_id = None
            try:
                user_id = payload["result"][0]["user_id"]
            except (KeyError, IndexError):
                return

            if url == "/user/state/mixer":
                result = payload["result"][0]
                log.info("/user/state/mixer {} {}, relay_url {}".format(
                    result.get('exist'), result['user_id'], result.get('mixer_url')))
                if result.get('exist', True):
                    mixer_url = result["mixer_url"]
                    log.info(
                        "/user/state/mixer exist == True, mixer_url {}".format(mixer_url))
                    if result.get('env', "dev") != config.ENV:
                        log.info("/user/state/mixer skipping create_sesison b/c our env {} != {}".format(
                            result.get('env', 'dev'), config.ENV))
                        return
                    self.create_session(user_id, mixer_url=mixer_url)
                else:
                    self.delete_session(user_id)
                return
            elif url == "/user/state/walle":
                result = payload["result"][0]
                log.info("/user/state/walle {} {}, relay_url {}".format(
                    result.get('exist'), result['user_id'], result.get('relay_url')))
                if result.get('exist', True):
                    relay_url = result.get("relay_url")
                    result_env = result.get('env', 'dev')
                    result_region = result.get('region', None)

                    if not relay_url:
                        log.info(
                            "/user/state/walle skipping create_sesison no relay_url {}".format(user_id))
                        return

                    if result_env != config.ENV or result_region != config.BEBO_REGION:
                        log.info("/user/state/walle skipping create_sesison env {} != {} or region {} != {}".format(
                            result_env, config.ENV, result_region, config.BEBO_REGION))
                        return
                    self.create_session(user_id, relay_url=relay_url)
                else:
                    self.delete_session(user_id)
            elif url == "/user/game":
                self.on_user_game(payload)

        except Exception:
            sentry_sdk.capture_exception()
            log.exception("Error handling ws payload {}".format(url))
            log.info("message %s", payload)

    def on_session_timeout(self, user_id):
        log.warning("on_session_timeout: %s", user_id)
        self.delete_session(user_id)

    def get_session_dot(self, user_id):
        session = self.session_map.get(user_id, None)
        if not session:
            return None

        return session.draw_dot()

    def delete_session(self, user_id):
        log.info("Deleting session: {}".format(user_id))

        if user_id in self.session_map:
            try:
                self.session_map[user_id].stop()
                del self.session_map[user_id]
            except KeyError:
                sentry_sdk.capture_exception()
                log.exception(
                    "Key error when deleteing session: {}".format(user_id))

                PharahAPI.delete_user_state_bran(user_id)

        lock = self.get_or_create_lock(user_id)
        if lock._held:
            lock.release()
            util.track_event({
                "label": "lock.release",
                "user_id": user_id,
            })
        else:
            log.info(
                "Deleting session doesn't hold the lock, handled by other host {}".format(user_id))

    def get_number_of_sessions(self):
        return len(self.session_map)

    def get_health(self):
        load = self.get_number_of_sessions()
        payload = {}
        payload["full"] = load >= config.CAPACITY
        payload["stoppable"] = False
        payload["stoppable"] = load < math.ceil(0.6 * config.CAPACITY)
        payload["capacity"] = config.CAPACITY
        payload["load"] = load
        payload["healthy"] = True
        return payload

    def stop(self):
        for session in list(self.session_map.keys()):
            self.delete_session(session)
