import time
import uuid
import threading
import random

import sentry_sdk

from bran.api.pharah import PharahAPI

import bran.shared.util as util
import bran.shared.image_utils as ImageUtils
from bran.shared.logger import log
from bran.shared.config import config


class BaseDetector(object):
    def __init__(self, user_id, **kwargs):
        self.name = kwargs['name']
        self.save_event = kwargs.get('save_event', False)
        self.game = kwargs.get('game', 'Fortnite')
        self.user_id = user_id

    def do_detect(self, user_id, detection_id, frame, timestamp, **kwargs):
        raise NotImplementedError

    def emit_detection(self, user_id, detection_id, detection, image_url):
        t = threading.Thread(target=self.emit_detection_sync, args=(
            user_id, detection_id, detection, image_url))
        t.start()

    def emit_detection_sync(self, user_id, detection_id, detection, image_url):
        return PharahAPI.post_user_detection(user_id, self.name, detection_id, detection["label"], detection["prob"], image_url, self.game)

    def detect(self, user_id, frame, timestamp, frame_id, **kwargs):
        start_time = time.time()
        try:
            detection_id = uuid.uuid4().hex

            detection = self.do_detect(
                user_id, detection_id, frame, timestamp, **kwargs)
            # log.info("detection: {}".format(detection))
            detection['name'] = self.name
            detection['game'] = self.game
            detection["detection_id"] = detection_id
            detection["timestamp"] = timestamp

            image_url = None
            if detection.get('detected', False):
                image_url = ImageUtils.save_preclassify_image(
                    frame, detection_id, self.game, self.name, detection['label'])
                log.info("DETECTED: {}, {}".format(detection, image_url))
                self.emit_detection(user_id, detection_id,
                                    detection, image_url)
            elif detection.get('duplicate', False):
                if random.random() < config.PRECLASSIFY_PERCENT / 100:
                    image_url = ImageUtils.save_preclassify_image(
                        frame, detection_id, self.game, self.name, detection['label'])
            elif random.random() < config.PRECLASSIFY_PERCENT / 100 and not ImageUtils.is_black_frame(frame):
                image_url = ImageUtils.save_preclassify_image(
                    frame, detection_id, self.game, self.name, "no")

            end_time = time.time()
            delta_time = (end_time - start_time) * 1000.0
            detection_bool = detection.get(
                "detected", False) or detection.get("duplicate", False)
            detection_prob = detection.get('prob', 0)

            compare_data = {
                self.name: detection_bool,
                '{}_prob'.format(self.name): detection_prob,
                '{}_time'.format(self.name): delta_time,
                "raw_image_location": image_url,
                "label": "detection_combo",
                "user_id": user_id,
                "frame_id": frame_id,
            }
            util.track_event(compare_data, frame_id)

            return detection
        except Exception as e:
            sentry_sdk.capture_exception()
            log.exception("failed to detect: {}".format(e))
