import cv2
import numpy as np

from bran.detectors.base_detector import BaseDetector
from bran.models.pytorch_resnet import PyTorchResNet
from bran.shared.logger import log
from bran.shared.util import S_TO_NS
import bran.shared.image_utils as ImageUtils

class BlackOp4KillDetector(BaseDetector):
    def __init__(self, *args, **kwargs):
        kwargs['game'] = 'blackops4'
        kwargs['name'] = 'blackops4_kill'
        kwargs['save_event'] = False
        super().__init__(*args, **kwargs)

        self.model = PyTorchResNet({
            "arch": "resnet18_pretrained",
            "n_classes": 2,
            "labels": ["no", "yes"],
            "weights": 'blackop4_kill/weights.pt'
        })

        self.dedupe_timestamp = None
        self.dedupe_time_threshold = 1.8

    def do_detect(self, user_id, detection_id, frame, timestamp, **kwargs):
        cropped_img = ImageUtils.crop(frame,
            { "x": 0.30, "y": 0.65, "width": 0.37, "height": 0.25 })
        image = cv2.resize(cropped_img, (224, 224))

        label, prob = self.model.detect(image)
        detected_prob = prob * 100

        detected = False
        duplicate = False

        if label == "yes" and detected_prob > 90:
            if self.dedupe_timestamp and (timestamp - self.dedupe_timestamp) < self.dedupe_time_threshold * S_TO_NS:
                log.info("DUPLICATE {} {:.3f}s -- IGNORING".format(self.name,
                                                                   (timestamp - self.dedupe_timestamp) / S_TO_NS))
                duplicate = True
            else:
                self.dedupe_timestamp = timestamp
                detected = True

        return {"detected": detected, "duplicate": duplicate, "prob": detected_prob, "label": label}
