import multiprocessing
import threading
import cachetools
import cv2
import numpy as np
from bran.detectors.fortnite_kill_detector import KillDetector
from bran.detectors.fortnite_victory_detector import VictoryDetector
from bran.detectors.fortnite_gamestate_detector import GameStateDetector
from bran.detectors.apex_kill_detector import ApexKillDetector
from bran.detectors.apex_victory_detector import ApexVictoryDetector
from bran.shared.logger import log
import sentry_sdk

DETECTORS = {
    "apex": [ApexKillDetector, ApexVictoryDetector],
    "fortnite": [KillDetector, VictoryDetector]  # GameStateDetector
}


class DetectorWorker(threading.Thread):
    def __init__(self, queue, detector_controller):
        super().__init__(daemon=True)
        self.queue = queue
        self.user_id = detector_controller.session.user_id
        self.game_id = detector_controller.session.match['game_id']

    def run(self):
        detectors = [detector(self.user_id)
                     for detector in DETECTORS[self.game_id]]
        for d in detectors:
            log.info("loading detector {}.{}".format(d.game, d.name))
        while True:
            try:
                item = self.queue.get()
                if item == 'STOP':
                    log.warn(
                        '%s Detector worker exiting because of STOP from Detector Queue', self.user_id)
                    break
                game_frame, timestamp, frame_id, user_game_id = item
                with sentry_sdk.push_scope() as scope:
                    scope.user = {"id": self.user_id}
                    for detector in detectors:
                        detector.detect(self.user_id, game_frame, timestamp,
                                        frame_id, user_game_id=user_game_id)
            except KeyboardInterrupt:
                raise
            except Exception as e:
                sentry_sdk.capture_exception()
                log.error("{} Uncaught exception: {}".format(
                    self.user_id, e), exc_info=True)
        log.info("%s Detector worker stopped running", self.user_id)
