import subprocess
from subprocess import check_call
from threading import Timer
import datetime
import re
import uuid

from gi.repository import Gst
import numpy as np
import sentry_sdk

from bran.controllers.detector_controller import DetectorController
from bran.shared import gst_util
from bran.shared import util
from bran.shared.logger import log


TIMEOUT_SECONDS = 30


class Session(object):
    def __init__(self, user_id, match, relay_url=None, mixer_url=None):
        self.detector_controller = None
        self.relay_url = relay_url
        self.mixer_url = mixer_url

        self.user_id = user_id
        self.match_id = match.get('match_id')
        self.match = match
        log.info("got match id: {} game_id: {}".format(
            self.match_id, match['game_id']))

        self.user_game_id = None

        self.src_tee = None

        self.started = False
        self.pipeline = None
        self.last_handoff_time = datetime.datetime.now()
        self.has_new_frame = False
        self.video_timeout_thread = None
        self.demux = None
        self.bus_message_connection = None
        self.pad_added_connection = None
        self.handoff_connection = None
        self.fakesink = None
        self.bus = None

        self.name = '[{}]'.format(self.user_id)

        self.req_id_map = {}

    def on_user_game(self, user_game_id):
        self.user_game_id = user_game_id

    def start(self):
        if self.started:
            log.error("Session already started")
            return
        self.started = True

        log.info("{} session start".format(self.name))

        self.pipeline = Gst.Pipeline.new("gamesense")
        self.bus = self.pipeline.get_bus()
        self.bus.add_signal_watch()
        self.bus_message_connection = self.bus.connect(
            "message", self.on_message)

        src = None
        if self.mixer_url is not None:
            log.info("{} mixer_url {}".format(self.mixer_url, self.name))
            src = Gst.ElementFactory.make("souphttpsrc")
            src.set_property("location", self.mixer_url)
        else:
            src = Gst.ElementFactory.make("rtmpsrc")
            src.set_property("location", self.relay_url)

        self.src_tee = Gst.ElementFactory.make("tee")

        # !important decodebin need this
        queue = Gst.ElementFactory.make("queue")
        decodebin = Gst.ElementFactory.make("decodebin")

        self.pipeline.add(src)
        self.pipeline.add(self.src_tee)
        self.pipeline.add(queue)
        self.pipeline.add(decodebin)

        gst_util.link_many(src, self.src_tee, queue, decodebin)

        self.demux = decodebin
        self.pad_added_connection = self.demux.connect(
            "pad-added", self.demux_pad_added)

        self.detector_controller = DetectorController(
            session=self,
            name=self.name)
        self.detector_controller.start()

        # cmd = [ 'ffmpeg', '-hide_banner', '-loglevel', 'warning',
        #         '-c:v', 'h264_cuvid',
        #         '-i', self.relay_url,
        #         '-f', 'image2pipe', '-vcodec', 'rawvideo', '-vf', 'fps=1,scale=1280:720', '-pix_fmt', 'rgb24', '-' ]
        # self.ffmpeg = subprocess.Popen(cmd, stdout=subprocess.PIPE, bufsize=10**8)

        # self.t = threading.Thread(target=self.loop)
        # self.t.daemon = True
        # self.t.start()

        self.pipeline.set_state(Gst.State.PLAYING)
        self.has_new_frame = True
        self.start_timeout_check()

    def stop(self):
        self.started = False
        if self.video_timeout_thread:
            self.video_timeout_thread.cancel()

        if self.pad_added_connection:
            self.demux.disconnect(self.pad_added_connection)
            self.pad_added_connection = None

        if self.handoff_connection:
            self.fakesink.disconnect(self.handoff_connection)
            self.handoff_connection = None

        if self.pipeline:
            self.bus.disconnect(self.bus_message_connection)
            self.bus.remove_signal_watch()
            self.pipeline.set_state(Gst.State.NULL)

        if self.detector_controller:
            self.detector_controller.stop()

        self.detector_controller = None
        self.pipeline = None
        self.bus = None
        self.bus_message_connection = None
        self.src_tee = None
        self.has_new_frame = False
        self.video_timeout_thread = None
        self.demux = None
        self.fakesink = None
        self.handoff_connection = None

    def on_message(self, _, message):
        t = message.type

        try:
            if t == Gst.MessageType.INFO:
                log.info("{}  bran gst Info on_message: {}".format(
                    self.name,
                    message.parse_info()))
            elif t == Gst.MessageType.WARNING:
                log.warning("{} bran gst Warning on_message: {}".format(
                    self.name,
                    message.parse_warning()))
            elif t == Gst.MessageType.ERROR:
                err, debug = message.parse_error()
                log.warning("{} bran gst Error on_message: {}: {}".format(
                    self.name,
                    err, debug))
                with sentry_sdk.push_scope() as scope:
                    scope.user = {"id": self.name}
                    sentry_sdk.capture_event({"message": "Bran Gst Error: {}".format(
                        err), "level": "error"}, hint={"debug": debug})
            elif t == Gst.MessageType.APPLICATION:
                pass
            elif t == Gst.MessageType.ELEMENT:
                pass
        except Exception as e:
            sentry_sdk.capture_exception()
            log.error("{} bran gst exception: %s", self.name, e, exc_info=True)

    def link_audio_pad(self, pad):
        log.info("{} linking audio pad".format(self.name))

        queue = Gst.ElementFactory.make("queue")
        queue.set_property("leaky", 2)
        audio_sink = Gst.ElementFactory.make("fakesink")

        self.pipeline.add(queue)
        self.pipeline.add(audio_sink)

        pad.link(queue.get_static_pad("sink"))
        gst_util.link_many(queue, audio_sink)

        queue.set_state(Gst.State.PLAYING)
        audio_sink.set_state(Gst.State.PLAYING)

    def link_video_pad(self, pad):
        log.info("{} linking video pad".format(self.name))

        videorate = Gst.ElementFactory.make("videorate")
        videoconvert = Gst.ElementFactory.make("videoconvert")
        capsfilter = gst_util.create_capsfilter(
            "video/x-raw,framerate=1/1,format=BGR")

        fakesink_queue = Gst.ElementFactory.make("queue")
        fakesink_queue.set_property("leaky", 2)

        fakesink = Gst.ElementFactory.make("fakesink")
        fakesink.set_property("signal-handoffs", True)
        self.fakesink = fakesink
        self.handoff_connection = fakesink.connect(
            "handoff", self.fakesink_handoff)

        elements = [videorate, videoconvert,
                    capsfilter, fakesink_queue, fakesink]

        for e in elements:
            self.pipeline.add(e)

        pad.link(videorate.get_static_pad("sink"))
        gst_util.link_many(*elements)

        for e in elements:
            e.set_state(Gst.State.PLAYING)

    def demux_pad_added(self, element, pad):
        if element != self.demux:
            return
        caps_string = pad.get_current_caps().to_string()
        log.info("{} demux_pad_added {}".format(self.name, caps_string))
        if "video" in caps_string:
            self.link_video_pad(pad)
        elif "audio" in caps_string:
            self.link_audio_pad(pad)

    def fakesink_handoff(self, _, buf, pad):
        try:
            self.has_new_frame = True
            self.last_handoff_time = datetime.datetime.now()

            caps = pad.get_current_caps()
            caps_structure = caps.get_structure(0)
            width = caps_structure.get_int("width").value
            height = caps_structure.get_int("height").value

            (_result, mapinfo) = buf.map(Gst.MapFlags.READ)
            nparr = np.fromstring(
                mapinfo.data, np.uint8).reshape(height, width, 3)
            buf.unmap(mapinfo)
            timestamp = buf.pts

            if self.started:
                self.detector_controller.new_frame(nparr, timestamp)
        except ValueError:
            log.warning("{} ValueError when processing frame".format(
                self.name), exc_info=True)
            util.track_event({
                "user_id": self.user_id,
                "label": "error",
                "error": "ValueError when processing frame"
            })
        except:
            log.exception("{} Failed to process frame".format(self.name))
            sentry_sdk.capture_exception()

    def start_timeout_check(self):
        if self.has_new_frame:
            log.info("{} timeout_check passed -- had frame".format(self.name))
            self.has_new_frame = False
            self.video_timeout_thread = Timer(
                TIMEOUT_SECONDS, self.start_timeout_check)
            self.video_timeout_thread.daemon = True
            self.video_timeout_thread.start()
            return

        log.info("{} timeout_check failed -- no new frame".format(self.name))
        util.track_event({
            "user_id": self.user_id,
            "label": "error",
            "error": "timeout",
            "relay_url": self.relay_url
        })

        self.on_timeout(self.user_id)

    def on_timeout(self, user_id):
        raise NotImplementedError

    def draw_dot(self):
        if self.pipeline is None:
            log.error("{} No pipeline when asking for dot".format(self.name))
            return ""

        log.info("{} *** DRAWING DOT ***".format(self.name))
        Gst.debug_bin_to_dot_file(
            self.pipeline, Gst.DebugGraphDetails.ALL, "get_dot")
        check_call(['dot', '-Tpng', '/tmp/get_dot.dot',
                    '-o', '/tmp/get_dot.png'])
        return "/tmp/get_dot.png"
