import datetime
import shutil
import time
import uuid
import os

from threading import Thread

import boto3
import cv2
import numpy as np

import bran.shared.util as util
from bran.shared.config import config
from bran.shared.logger import log

s3 = boto3.client("s3")


def crop(image, crop_box):
    image_height, image_width, _ = image.shape
    left = int(crop_box["x"] * image_width)
    width = int(crop_box["width"] * image_width)
    top = int(crop_box["y"] * image_height)
    height = int(crop_box["height"] * image_height)

    left_padding = 0
    right_padding = 0
    top_padding = 0
    bottom_padding = 0

    if left < 0:
        left_padding = left * -1
        left = 0

    if top < 0:
        top_padding = top * -1
        top = 0

    if (width + left_padding + left) > image_width:
        right_padding = (width + left_padding + left) - image_width

    if (height + top_padding + top) > image_height:
        bottom_padding = (height + top_padding + top) - image_height

    padded_image = image
    if left_padding != 0 or right_padding != 0 or top_padding != 0 or bottom_padding != 0:
        padded_image = cv2.copyMakeBorder(
            image, top_padding, bottom_padding, left_padding, right_padding, cv2.BORDER_REFLECT)

    return padded_image[top:top+height, left:left+width]


def bgr_to_hsv(image):
    return cv2.cvtColor(image, cv2.COLOR_BGR2HSV)


def save_image_dispatcher(img):
    image_name = "{}.png".format(uuid.uuid4().hex)

    pickup_time = int(round((time.time()) / 60)) * 60
    pickup_date = datetime.datetime.fromtimestamp(pickup_time)

    year = pickup_date.year
    month = pickup_date.month
    day = pickup_date.day
    hour = pickup_date.hour
    image_url = "{}/{}/{}/{}/{}/{}/{}-{}".format(config.DISPATCHER_BUCKET_URL, "gamesense_media",
                                                 year, month, day, hour, config.HOSTNAME.split(".")[0], image_name)
    image_location = "/var/bran/gamesense_media/{}".format(image_name)
    dispatcher_image_location = "{}/gamesense_media/{}__{}".format(
        config.DISPATCHER_BASE_DIR, image_name, pickup_time)

    def save_image(image, image_location, dispatcher_image_location):
        cv2.imwrite(image_location, image)
        shutil.move(image_location, dispatcher_image_location)

    t = Thread(target=save_image, args=(
        img, image_location, dispatcher_image_location))
    t.start()

    return image_url


def save_image_s3(image, bucket, name):
    def s3_upload():
        image_name = uuid.uuid4().hex
        filename = "{}/{}.png".format(config.TEMP_IMAGE_DIR, image_name)
        cv2.imwrite(filename, image)
        s3.upload_file(filename, bucket, name)

    t = Thread(target=s3_upload)
    t.start()


def save_preclassify_image(image, detection_id, game, detection_type, classification):
    if config.ENV != 'bebo-prod' and config.ENV !='dev':
        pass

    destination = "{}_{}_{}".format(game, detection_type, classification)
    filename = "{}/{}_preclassify.png".format(
        config.TEMP_IMAGE_DIR, detection_id)
    name = "pre_classify/{}/{}.png".format(
        destination, detection_id)
    image_url = "{}/{}/{}".format(config.GAMESENSE_BUCKET_URL,
                                  config.GAMESENSE_BUCKET, name)

    def s3_upload():
        cv2.imwrite(filename, image)
        log.info('Uploading pre_classify image {} to {} from {}'.format(name, config.GAMESENSE_BUCKET, filename))

        s3.upload_file(filename, config.GAMESENSE_BUCKET, name)
        os.remove(filename)
    t = Thread(target=s3_upload)
    t.start()
    util.track_event({
        "category": "pre_classify",
        "action": "upload",
        "label": destination,
        "image_url": image_url,
        "detection_id": detection_id,
        "game": game,
        "detection_type": detection_type
    })
    return image_url

def is_black_frame(frame):
    return np.sum(frame) < 1
