package server

import (
	"context"
	"errors"
	"fmt"

	"code.justin.tv/esports-exp/centaur/internal/redrec"
	"code.justin.tv/esports-exp/centaur/internal/storage"
	centaur "code.justin.tv/esports-exp/centaur/proto"
	"github.com/twitchtv/twirp"
)

func (s *Centaur) AddEmote(ctx context.Context, req *centaur.AddEmoteRequest) (*centaur.AddEmoteResponse, error) {

	if req == nil {
		return nil, twirp.NewError(twirp.Internal, "No req")
	}

	if req.DeviceId == "" {
		return nil, twirp.RequiredArgumentError("device_id")
	}

	if req.ClipId == "" {
		return nil, twirp.RequiredArgumentError("clip_id")
	}

	user, err := s.gorm.GetUserByDeviceID(req.DeviceId)
	if err != nil {
		return nil, twirp.NewError(twirp.Internal, err.Error())
	}

	clip, err := s.gorm.GetClipByClipID(req.ClipId)
	if err != nil {
		return nil, twirp.NewError(twirp.Internal, err.Error())
	}

	reaction := &storage.Reaction{
		UserID:  user.ID.String(),
		ClipID:  clip.ID,
		EmoteID: req.EmoteId,
		Amount:  1,
	}

	if err = reaction.Add(s.gorm); err != nil {
		return nil, twirp.InternalError(err.Error())
	}

	rr := redrec.NewFromConn(s.rp.Get())
	if err = rr.Rate(clip.ID.String(), user.ID.String(), 1.0); err != nil {
		return nil, twirp.NewError(twirp.Internal, err.Error())
	}

	return &centaur.AddEmoteResponse{}, nil
}

func (s *Centaur) ReportTimeWatched(ctx context.Context, req *centaur.ReportTimeWatchedRequest) (*centaur.ReportTimeWatchedResponse, error) {
	if req == nil {
		return nil, twirp.NewError(twirp.Internal, "No req")
	}

	if req.DeviceId == "" {
		return nil, twirp.RequiredArgumentError("device_id")
	}

	if req.ClipId == "" {
		return nil, twirp.RequiredArgumentError("clip_id")
	}

	user, err := s.gorm.GetUserByDeviceID(req.DeviceId)
	if err != nil {
		return nil, twirp.NewError(twirp.Internal, err.Error())
	}

	clip, err := s.gorm.GetClipByClipID(req.ClipId)
	if err != nil {
		return nil, twirp.NewError(twirp.Internal, err.Error())
	}

	score, err := s.getMaxScore(ctx, user, clip, calcWatchedScore(req.Duration, clip.Duration))
	if err != nil {
		return nil, twirp.NewError(twirp.Internal, err.Error())
	}

	fmt.Printf("User rate %s %s %f, [%d/%d]\n", user.ID, clip.ID, score, req.Duration, clip.Duration)

	rr := redrec.NewFromConn(s.rp.Get())
	if err = rr.Rate(clip.ID.String(), user.ID.String(), score); err != nil {
		return nil, twirp.NewError(twirp.Internal, err.Error())
	}

	if err = s.gorm.MarkClipWatchedByUser(clip, user, req.Duration); err != nil {
		return nil, twirp.NewError(twirp.Internal, err.Error())
	}

	return &centaur.ReportTimeWatchedResponse{}, nil
}

func (s *Centaur) ReportShared(ctx context.Context, req *centaur.ReportSharedRequest) (*centaur.ReportSharedResponse, error) {
	return nil, errors.New("not implemented")
}
