package server

import (
	"context"
	"fmt"
	"math"
	"time"

	"code.justin.tv/esports-exp/centaur/internal/config"
	"code.justin.tv/esports-exp/centaur/internal/redrec"
	"code.justin.tv/esports-exp/centaur/internal/storage"
	centaur "code.justin.tv/esports-exp/centaur/proto"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/gomodule/redigo/redis"
	"github.com/twitchtv/twirp"
)

type Centaur struct {
	gorm *storage.Gorm
	rp   *redis.Pool

	s3 *s3.S3
}

func NewCentaur() *Centaur {
	cfg := config.New()

	gorm, err := storage.New(cfg)
	if err != nil {
		panic(fmt.Errorf("failed to setup gorm: %s", err))
	}

	rp := &redis.Pool{
		MaxIdle:     3,
		IdleTimeout: 240 * time.Second,
		// Dial or DialContext must be set. When both are set, DialContext takes precedence over Dial.
		Dial: func() (redis.Conn, error) { return redis.DialURL(cfg.RedisHost) },
	}

	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-1")},
	)

	if err != nil {
		panic(fmt.Errorf("failed to create AWS session: %s", err.Error()))
	}

	s3Svc := s3.New(sess)

	return &Centaur{
		gorm,
		rp,

		s3Svc,
	}
}

func (s *Centaur) GetClips(ctx context.Context, req *centaur.GetClipsRequest) (*centaur.GetClipsResponse, error) {
	if req == nil {
		return nil, twirp.NewError(twirp.Internal, "No req")
	}

	if req.DeviceId == "" {
		return nil, twirp.RequiredArgumentError("device_id")
	}

	if req.Limit == 0 {
		req.Limit = 10
	}

	user, err := s.gorm.GetUserByDeviceID(req.DeviceId)
	if err != nil {
		return nil, twirp.NewError(twirp.Internal, err.Error())
	}

	recCnt, randomCnt, err := s.getSrcDistribution(user, req.Limit)
	if err != nil {
		return nil, twirp.NewError(twirp.Internal, err.Error())
	}

	fmt.Printf("Serving %d/%d from recs, and %d/%d from random \n", recCnt, req.Limit, randomCnt, req.Limit)

	var clipIDs []string
	if recCnt > 0 {
		rr := redrec.NewFromConn(s.rp.Get())
		rr.UpdateSuggestedItems(user.ID.String(), int(recCnt)-1)
		clipRecs, err := rr.GetUserSuggestions(user.ID.String(), int(recCnt)-1)
		if err != nil {
			return nil, twirp.NewError(twirp.Internal, err.Error())
		}

		for i, clip := range clipRecs {
			if i%2 == 0 {
				clipIDs = append(clipIDs, clip)
			}
		}
	}

	recClips, err := s.gorm.GetClipsFromClipIDS(clipIDs)
	if err != nil {
		return nil, twirp.NewError(twirp.Internal, err.Error())
	}

	recCntReturned := int64(len(recClips))

	if recCntReturned < recCnt {
		diff := recCnt - recCntReturned
		fmt.Printf("%d/%d recs fulfilled -- adding %d to random \n", recCntReturned, recCnt, diff)
		randomCnt += diff
	}

	// randClips, err := user.GetRandomNewClips(s.gorm, randomCnt
	randClips, err := user.GetRandomWeightedClips(s.gorm, randomCnt)
	if err != nil {
		return nil, twirp.NewError(twirp.Internal, err.Error())
	}

	var out []*centaur.Clip
	for _, clip := range randClips {
		if err := clip.LoadReactions(s.gorm); err != nil {
			return nil, err
		}
		cc := clip.ToProto()
		cc.RecType = "random"
		out = append(out, cc)
	}

	for _, clip := range recClips {
		if err := clip.LoadReactions(s.gorm); err != nil {
			return nil, err
		}
		cc := clip.ToProto()
		cc.RecType = "rec"
		out = append(out, cc)
		if len(out) == int(req.Limit) {
			break
		}
	}

	return &centaur.GetClipsResponse{
		Clips: out,
	}, nil
}

func (s *Centaur) getSrcDistribution(user *storage.User, limit int64) (int64, int64, error) {
	cnt, err := user.CountWatchedClips(s.gorm)
	if err != nil {
		return 0, 0, err
	}

	fmt.Printf("User has watched %d clips \n", cnt)

	recRatio := float64(0)
	if cnt < 10 {
		recRatio = 0.2
	} else if cnt < 20 {
		recRatio = 0.4
	} else if cnt < 30 {
		recRatio = 0.6
	} else if cnt < 40 {
		recRatio = 0.8
	} else {
		recRatio = 0.9
	}

	// at least 1 random
	recCnt := int64(math.Floor(recRatio * float64(limit)))
	return recCnt, limit - recCnt, nil
}
